package io.ultreia.java4all.lang;

/*-
 * #%L
 * Java Lang extends by Ultreia.io
 * %%
 * Copyright (C) 2017 - 2018 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.MessageFormat;
import java.util.Locale;

/**
 * Created by tchemit on 29/12/2017.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class Strings {
    public static final String[] EMPTY_STRING_ARRAY = new String[0];
    private static final double[] timeFactors = {1000000, 1000, 60, 60, 24};
    private static final String[] timeUnites = {"ns", "ms", "s", "m", "h", "d"};
    private static final double[] memoryFactors = {1024, 1024, 1024, 1024};
    private static final String[] memoryUnites = {"o", "Ko", "Mo", "Go", "To"};
    private static final char[] HEX_CHARS = {
            '0', '1', '2', '3',
            '4', '5', '6', '7',
            '8', '9', 'a', 'b',
            'c', 'd', 'e', 'f',};

    /**
     * Convert a String to SHA1.
     *
     * @param toEncode string to encode
     * @return sha1 corresponding
     * @throws IllegalStateException if could not found algorithm SHA1
     */
    public static String encodeSHA1(String toEncode) {

        try {
            MessageDigest sha1Md = MessageDigest.getInstance("SHA-1");
            byte[] digest = sha1Md.digest(toEncode.getBytes());
            return asHex(digest);
        } catch (NoSuchAlgorithmException ex) {
            throw new IllegalStateException("Can't find SHA-1 message digest algorithm", ex);
        }

    }

    /**
     * Turns array of bytes into string representing each byte as
     * unsigned hex number.
     *
     * @param hash Array of bytes to convert to hex-string
     * @return Generated hex string
     */
    private static String asHex(byte hash[]) {
        char buf[] = new char[hash.length * 2];
        for (int i = 0, x = 0; i < hash.length; i++) {
            buf[x++] = HEX_CHARS[hash[i] >>> 4 & 0xf];
            buf[x++] = HEX_CHARS[hash[i] & 0xf];
        }
        return new String(buf);
    }

    /**
     * Converts an time delay into a human readable format.
     *
     * @param value the delay to convert
     * @return the memory representation of the given value
     * @see #convert(long, double[], String[])
     */
    public static String convertTime(long value) {
        return convert(value, timeFactors, timeUnites);
    }

    /**
     * Converts an time period into a human readable format.
     *
     * @param value  the begin time
     * @param value2 the end time
     * @return the time representation of the given value
     * @see #convert(long, double[], String[])
     */
    public static String convertTime(long value, long value2) {
        return convertTime(value2 - value);
    }

    /**
     * Converts an memory measure into a human readable format.
     *
     * @param value the memory measure to convert
     * @return the memory representation of the given value
     * @see #convert(long, double[], String[])
     */
    public static String convertMemory(long value) {
        return convert(value, memoryFactors, memoryUnites);
    }

    /**
     * Note: this method use the current locale (the {@link Locale#getDefault()}) in the method
     * {@link MessageFormat#MessageFormat(String)}.
     *
     * @param value   value to convert
     * @param factors facotrs used form conversion
     * @param unites  libelle of unites to use
     * @return the converted representation of the given value
     */
    public static String convert(long value, double[] factors, String[] unites) {
        long sign = value == 0 ? 1 : value / Math.abs(value);
        int i = 0;
        double tmp = Math.abs(value);
        while (i < factors.length && i < unites.length && tmp > factors[i]) {
            tmp = tmp / factors[i++];
        }

        tmp *= sign;
        String result;
        result = MessageFormat.format("{0,number,0.###}{1}", tmp, unites[i]);
        return result;
    }

    /**
     * Convertir un nom en une constante Java
     * <p>
     * Les seuls caractères autorisés sont les alpha numériques, ains
     * que l'underscore. tous les autres caractères seront ignorés.
     *
     * @param name le nom à convertir
     * @return la constante générée
     */
    public static String convertToConstantName(String name) {
        StringBuilder sb = new StringBuilder();
        char lastChar = 0;
        for (int i = 0, j = name.length(); i < j; i++) {
            char c = name.charAt(i);
            if (Character.isDigit(c)) {
                sb.append(c);
                lastChar = c;
                continue;
            }
            if (!Character.isLetter(c)) {
                if (lastChar != '_') {
                    sb.append('_');
                }
                lastChar = '_';
                continue;
            }
            if (Character.isUpperCase(c)) {
                if (!Character.isUpperCase(lastChar) && lastChar != '_') {
                    sb.append('_');
                }
                sb.append(c);
            } else {
                sb.append(Character.toUpperCase(c));
            }
            lastChar = c;
        }
        String result = sb.toString();
        // clean tail
        while (!result.isEmpty() && result.endsWith("_")) {
            result = result.substring(0, result.length() - 1);
        }
        // clean head
        while (!result.isEmpty() && result.startsWith("_")) {
            result = result.substring(1);
        }
        return result;
    }
}
