package io.ultreia.java4all.lang;

/*-
 * #%L
 * Java Lang extends by Ultreia.io
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * Created by tchemit on 03/10/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
//@SuppressWarnings("unchecked")
public class Objects2 {

    public static <O> O newInstance(Class<O> type) {
        Objects.requireNonNull(type);
        try {
            return type.newInstance();
        } catch (IllegalAccessException | InstantiationException e) {
            throw new IllegalStateException(String.format("Can't instantiate type: %s", type.getName()), e);
        }
    }

    public static void copy(GetterProducer source, SetterProducer target) {
        TypedGetterProducer typedSource = (TypedGetterProducer) source;
        Map<String, Function> getters = typedSource.getters();
        TypedSetterProducer typedTarget = ((TypedSetterProducer) target);
        Map<String, BiConsumer> setters = typedTarget.setters();
        copy(typedSource, typedTarget, getters, setters);
    }

    public static void copy(TypedGetterProducer source, TypedSetterProducer target) {
        copy(source, target, source.getters(), target.setters());
    }

    public static <D extends SetterProducer> void clear(D bean) {
        clear((TypedSetterProducer) bean);
    }

    public static <D extends TypedSetterProducer> void clear(D bean) {
        Map<String, BiConsumer> setters = bean.setters();
        for (BiConsumer o : setters.values()) {

            try {
                o.accept(bean, null);
            } catch (Exception e) {
                o.accept(bean, 0);
            }
        }
    }

    private static void copy(TypedGetterProducer source, TypedSetterProducer target, Map<String, Function> getters, Map<String, BiConsumer> setters) {
        for (Map.Entry<String, Function> entry : getters.entrySet()) {
            String propertyName = entry.getKey();
            BiConsumer setter = setters.get(propertyName);
            if (setter != null) {
                Function getter = entry.getValue();
                Object propertyValue = getter.apply(source);
                setter.accept(target, propertyValue);
            }
        }
    }

    public static <T, O> O get(GetterProducer object, String propertyName, Map<String, Function<T, ?>> getters) {
        Function getter = getters.get(propertyName);
        Objects.requireNonNull(getter, String.format("Can't find getter for property %s on %s", propertyName, object.getClass().getName()));
        return (O) getter.apply(object);
    }

    public static <T, O> void set(SetterProducer object, String propertyName, O propertyValue, Map<String, BiConsumer<T, ?>> setters) {
        BiConsumer setter = setters.get(propertyName);
        Objects.requireNonNull(setter, String.format("Can't find a setter for property: %s on %s", propertyName, object.getClass().getName()));
        setter.accept(object, propertyValue);
    }

    static final Predicate<PropertyDescriptor> IS_READ_DESCRIPTOR = input -> input.getReadMethod() != null;
    static final Predicate<PropertyDescriptor> IS_WRITE_DESCRIPTOR = input -> input.getWriteMethod() != null;

    public static void walk(ClassWalkVisitor visitor, Class<?> beanType) {
        new ClassWalker(visitor).visit(beanType);
    }

    static class ClassWalker {

        private final ClassWalkVisitor visitor;
        private final Set<Class<?>> exploredTypes;

        ClassWalker(ClassWalkVisitor visitor) {
            this.visitor = visitor;
            this.exploredTypes = new LinkedHashSet<>();
        }

        public void visit(Class<?> beanType) {
            exploredTypes.clear();
            accept(beanType);
        }

        protected boolean accept(Class<?> beanType) {
            if (exploredTypes.contains(beanType)) {

                // already explored
                return visitor.canContinue();
            }
            exploredTypes.add(beanType);

            // get properties for the class
            visitor.onVisit(beanType);
            if (!visitor.canContinue()) {
                return false;
            }
            if (beanType.getSuperclass() != null) {

                // get properties fro super-class
                boolean canContinue = accept(beanType.getSuperclass());
                if (!canContinue) {
                    return false;
                }
            }

            for (Class<?> anInterface : beanType.getInterfaces()) {

                // get properties fro super-class
                boolean canContinue = accept(anInterface);
                if (!canContinue) {
                    return false;
                }
            }
            return visitor.canContinue();
        }
    }

    interface ClassWalkVisitor {

        void onVisit(Class<?> beanType);

        boolean canContinue();
    }

    private Objects2() {
    }
}
