package io.ultreia.java4all.lang;

/*-
 * #%L
 * Java Lang extends by Ultreia.io
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;
import java.util.function.Predicate;
import org.apache.commons.beanutils.PropertyUtils;

/**
 * Created by tchemit on 25/09/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class Setters {

    public static final Predicate<PropertyDescriptor> IS_WRITE_DESCRIPTOR = input -> input.getWriteMethod() != null;

    /**
     * Obtains all writeable properties from a given type.
     *
     * @param beanType the type to seek
     * @return the set of all writeable properties for the given type
     * @since 2.0
     */
    public static Set<String> getWriteableProperties(Class<?> beanType) {
        Set<Class<?>> exploredTypes = new HashSet<Class<?>>();
        Set<String> result = new HashSet<String>();

        // get properties for the class
        getWriteableProperties(beanType, result, exploredTypes);

        return result;
    }

    public static Method getMutator(Object bean, String property) {
        Method mutator = null;
        if (bean == null) {
            throw new NullPointerException("could not find bean");
        }
        if (property == null) {
            throw new NullPointerException("could not find property");
        }

        try {
            PropertyDescriptor descriptor = PropertyUtils.getPropertyDescriptor(bean, property);
            if (descriptor != null) {
                mutator = descriptor.getWriteMethod();
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        return mutator;
    }

    private static void getWriteableProperties(Class<?> beanType, Set<String> result, Set<Class<?>> exploredTypes) {

        if (exploredTypes.contains(beanType)) {

            // already explored
            return;
        }
        exploredTypes.add(beanType);

        // get properties for the class
        getWriteableProperties(beanType, result);

        if (beanType.getSuperclass() != null) {

            // get properties fro super-class
            getWriteableProperties(beanType.getSuperclass(), result, exploredTypes);
        }
        Class<?>[] interfaces = beanType.getInterfaces();
        for (Class<?> anInterface : interfaces) {

            // get properties fro super-class
            getWriteableProperties(anInterface, result, exploredTypes);
        }
    }

    private static void getWriteableProperties(Class<?> beanType, Set<String> result) {

        PropertyDescriptor[] descriptors = PropertyUtils.getPropertyDescriptors(beanType);
        for (PropertyDescriptor descriptor : descriptors) {
            String name = descriptor.getName();
            if (descriptor.getWriteMethod() != null) {
                result.add(name);
            }
        }
    }
}
