/*
 * #%L
 * EUGene :: EUGene Core
 * %%
 * Copyright (C) 2004 - 2017 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as 
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */
package org.nuiton.eugene.java;

import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.eugene.EugeneCoreTagValues;
import org.nuiton.eugene.GeneratorUtil;
import org.nuiton.eugene.models.object.ObjectModelAttribute;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelJavaModifier;
import org.nuiton.eugene.models.object.ObjectModelModifier;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugene.models.object.ObjectModelParameter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Utility class for pure java templates.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 2.0.2
 */
public class JavaGeneratorUtil extends GeneratorUtil {

    public static final String OPERATION_GETTER_DEFAULT_PREFIX = "get";

    public static final String OPERATION_GETTER_BOOLEAN_PREFIX = "is";

    /**
     * Dependency to add constants in interfaces via an enumeration.
     *
     * The literals of enumeration will be the value, and constant names will
     * be generated by transformers.
     *
     * @since 2.0.2
     */
    public static final String DEPENDENCIES_CONSTANTS = "constants";

    /**
     * Duplicates in the {@code target} classifier the given {@code operations}
     * using a {@code transformer} to modify model.
     *
     * @param transformer    the transformer to use
     * @param operations     operations to duplicate
     * @param target         where to duplicate operations
     * @param useVisibility  flag to use operation visibilty to compute his
     *                       modifiers
     * @param extraModifiers scopes to apply to all operations
     */
    public static void cloneOperations(ObjectModelTransformerToJava transformer,
                                       Iterable<ObjectModelOperation> operations,
                                       ObjectModelClassifier target,
                                       boolean useVisibility,
                                       ObjectModelModifier... extraModifiers) {

        for (ObjectModelOperation op : operations) {
            ObjectModelOperation resultOp;
            ObjectModelModifier[] modifiers;
            if (useVisibility) {

                // compute visibility modifer
                String visibility = op.getVisibility();
                ObjectModelModifier modifier =
                        ObjectModelJavaModifier.fromVisibility(visibility);

                int length = extraModifiers.length;
                if (length == 0) {
                    modifiers = new ObjectModelModifier[]{modifier};
                } else {
                    modifiers = new ObjectModelModifier[length + 1];
                    modifiers[0] = modifier;
                    System.arraycopy(extraModifiers, 0, modifiers, 1, length);
                }
            } else {

                // just use the incoming modifiers
                modifiers = extraModifiers;
            }

            resultOp = transformer.addOperation(target,
                                                op.getName(),
                                                op.getReturnType(),
                                                modifiers
            );
            for (ObjectModelParameter param : op.getParameters()) {
                transformer.addParameter(resultOp,
                                         param.getType(),
                                         param.getName()
                );
            }
            for (String exception : op.getExceptions()) {
                transformer.addException(resultOp, exception);
            }
        }
    }

    /**
     * Convertit une propriété javaBean son suffix utilisé pour écrire les
     * getter/setter correspondant.
     *
     * @param propertyName le nom de propriété a convertir
     * @return le nom capitalizé de la propriété JavaBean
     * @since 2.4.2
     */
    public static String capitalizeJavaBeanPropertyName(String propertyName) {

        if (StringUtils.isEmpty(propertyName)) {
            return propertyName;
        }

        if (propertyName.length() == 1) {

            // simple case :
            return propertyName.toUpperCase();
        }

        // check if second caracter is up
        char c = propertyName.charAt(1);
        boolean carIsUp = Character.isUpperCase(c);
        if (carIsUp) {

            // this is a special and strange case : must not capitalize first char
            return propertyName;
        }

        return StringUtils.capitalize(propertyName);
    }

    /**
     * Split the given fqn which contains a generic declaration into his parts.
     *
     * Example :
     * <pre>
     * java.util.List&lt;Integer&gt; : [java.util.List, Integer]
     * java.util.List&lt;java.util.List&lt;Integer&gt;&gt; : [java.util.List, java.util.List&lt;Integer&gt;]
     * java.util.Map&lt;Integer, Integer&gt; : [java.util.Map, Integer, Integer]
     * java.util.Map&lt;Integer, java.util.List&lt;Integer&gt;&gt; : [java.util.Map, Integer, java.util.List&lt;Integer&gt;]
     *
     * </pre>
     * <strong>Note:</strong> We only deal with one level of generics here in
     * order to be able easly to rebuild after all the fqn with simple names...
     *
     * @param fqn the fqn to split
     * @return the array of all parts of the fqn.
     * @since 2.3.2
     */
    public static String[] splitGeneric(String fqn) {
        if (fqn == null) {
            return null;
        }

        int firstPartIndex = fqn.indexOf('<');
        if (firstPartIndex == -1) {

            // no generics, just return the fqn
            return new String[]{fqn};
        }

        List<String> parts = new ArrayList<>();
        parts.add(fqn.substring(0, firstPartIndex));
        String rest = fqn.substring(firstPartIndex + 1,
                                    fqn.lastIndexOf('>'));

        if (containsGenerics(rest) && rest.contains(",")) {

            // there is others generics in generics, worse case...
            int begin = 0;
            int count = 0;
            int max = rest.length();
            for (int i = 0; i < max; i++) {
                char c = rest.charAt(i);
                if (c == '<') {

                    count++;
                } else if (c == '>') {

                    count--;
                } else if (c == ',') {

                    // arrives on a possible split
                    if (count == 0) {

                        // can safely add this part
                        String part = rest.substring(begin, i);
                        parts.add(part.trim());
                        begin = i + 1;
                    }
                }
            }

            // there is a last part to add ?
            if (begin < max) {
                String part = rest.substring(begin, max);
                parts.add(part.trim());
            }

        } else {

            // simple case : each part of the generics has no other generics
            String[] split = rest.split(",");
            for (String part : split) {
                parts.add(part.trim());
            }
        }

        return parts.toArray(new String[parts.size()]);
    }

    /**
     * Join generics parts of a fqn into aparts.
     *
     * Example :
     * <pre>
     * [java.util.List, Integer] : java.util.List&lt;Integer&gt;
     * [java.util.List, java.util.List&lt;Integer&gt;] : java.util.List&lt;java.util.List&lt;Integer&gt;&gt;
     * [java.util.Map, Integer, Integer] : java.util.Map&lt;Integer, Integer&gt;
     * [java.util.Map, Integer, java.util.List&lt;Integer&gt;] : java.util.Map&lt;Integer, java.util.List&lt;Integer&gt;&gt;
     * </pre>
     *
     * @param genericParts the parts of fqn
     * @return the fqn from his parts
     * @since 2.3.2
     */
    public static String joinGeneric(String... genericParts) {

        if (genericParts == null || genericParts.length == 0) {

            // this case should never happen ?
            return null;
        }

        if (genericParts.length == 1) {

            // in fact, no generics
            return genericParts[0];
        }

        StringBuilder sb = new StringBuilder(genericParts[0]);

        sb.append("<").append(genericParts[1]);

        for (int i = 2, max = genericParts.length; i < max; i++) {
            String genericPart = genericParts[i];
            sb.append(", ").append(genericPart);
        }
        sb.append('>');
        return sb.toString();
    }

    /**
     * Tells if the given fqn contains a generic declaration (says contains a
     * {@code <} caracter).
     *
     * @param fqn the fqn to test
     * @return {@code true} if given fqn contains a generic declaration
     * @since 2.3.2
     */
    public static boolean containsGenerics(String fqn) {
        return fqn.contains("<");
    }

    /**
     * Split the given fqns gieven the list separator.
     *
     * Example :
     * <pre>
     * Boolean, File : [Boolean, File]
     * Boolean , java.util.List&lt;Integer&gt; : [Boolean, java.util.List&lt;Integer&gt;]
     *
     * </pre>
     *
     * <strong>Note:</strong> You can NOT use as separator {@code '&lt;'} nor
     * {@code '&gt;'} nor {@code ' '}.
     *
     * @param fqns      the fqn list to split
     * @param separator the fqn separactor char
     * @return the array of all parts of the fqn.
     * @since 2.3.2
     */
    public static String[] splitFqnList(String fqns, char separator) {
        if (separator == '<' || separator == '>' || separator == ' ') {
            throw new IllegalArgumentException(
                    "Can not use '<' nor '>' for the separator");
        }
        if (fqns == null) {
            return null;
        }

        List<String> parts = new ArrayList<>();

        // there is others generics in generics, worse case...
        int begin = 0;
        int count = 0;
        int max = fqns.length();
        for (int i = 0; i < max; i++) {
            char c = fqns.charAt(i);
            if (c == '<') {

                count++;
            } else if (c == '>') {

                count--;
            } else if (c == separator) {

                // arrives on a possible split
                if (count == 0) {

                    // can safely add this part
                    String part = fqns.substring(begin, i);
                    parts.add(part.trim());
                    begin = i + 1;
                }
            }
        }

        // there is a last part to add ?
        if (begin < max) {
            String part = fqns.substring(begin, max);
            parts.add(part.trim());
        }

        return parts.toArray(new String[parts.size()]);
    }

    public static boolean isOrdered(ObjectModelAttribute attr) {
        return attr.isOrdered() || EugeneCoreTagValues.isOrdered(attr);
    }

    public static Class<?> getCollectionType(ObjectModelAttribute attr) {

        boolean ordered = isOrdered(attr);
        boolean unique = EugeneCoreTagValues.isUnique(attr);

        // Change type for Multiple attribute
        Class<?> result;

        if (ordered) {
            if (unique) {
                result = LinkedHashSet.class;
            } else {
                result = List.class;
            }
        } else {
            if (unique) {
                result = Set.class;
            } else {
                result = Collection.class;
            }
        }
        return result;
    }

    public static Class<?> getCollectionInstanceType(ObjectModelAttribute attr) {

        boolean ordered = isOrdered(attr);
        boolean unique = EugeneCoreTagValues.isUnique(attr);

        // Change type for Multiple attribute
        Class<?> result;

        if (ordered) {
            if (unique) {
                result = LinkedHashSet.class;
            } else {
                result = LinkedList.class;
            }
        } else {
            if (unique) {
                result = HashSet.class;
            } else {
                result = LinkedList.class;
            }
        }
        return result;
    }

    /**
     * Used to return the {@code operation} parameters for its declaration :
     * type and name of each parameter will be join as a string separated by a
     * comma. Usefull for operation parameters declaration in templates writing.
     *
     * @param operation to treate
     * @return the string corresponding to the list of operation parameters
     * for declaration syntax.
     */
    public static String getOperationParametersListDeclaration(ObjectModelOperation operation) {
        return Joiner.on(", ").join(operation.getParameters().stream().map(param->getAttributeInterfaceType(param) + ' ' + param.getName()).collect(Collectors.toList()));
    }

    /**
     * @param clazz the class where to look at
     * @return sous forme de String la liste des déclarations des attributes d'une classe donnée
     */
    public static String getClassAttributesListDeclaration(
            ObjectModelClass clazz) {
        StringBuilder result = new StringBuilder();

        Collection<ObjectModelAttribute> params = clazz.getAttributes();
        for (Iterator<ObjectModelAttribute> j = params.iterator(); j.hasNext(); ) {
            ObjectModelAttribute attr = j.next();
            result.append(getAttributeInterfaceType(attr, true)).append(" ").append(attr.getName());
            if (j.hasNext()) {
                result.append(", ");
            }
        }
        return result.toString();
    }

    /**
     * @param attribute the attribute to test
     * @return the type of the given attribute
     * @see #getAttributeInterfaceType(ObjectModelParameter, boolean)
     */
    public static String getAttributeInterfaceType(ObjectModelParameter attribute) {
        return getAttributeInterfaceType(attribute, false);
    }

    /**
     * Retourne le type de l'attribut, c-a-d une List ou une collection
     * ou le type defini si la cardinalité n'est pas multiple
     *
     * @param attribute   the attribute to test
     * @param useGenerics {@code true} if the attribute use a generic type
     * @return attribute type
     */
    public static String getAttributeInterfaceType(ObjectModelParameter attribute,
                                                   boolean useGenerics) {
        return getAttributeInterfaceType(attribute, attribute.getType(), useGenerics);
    }

    /**
     * Retourne le type de l'attribut, c-a-d une List ou une collection
     * ou le type defini si la cardinalité n'est pas multiple
     *
     * @param attribute     the attribute to test
     * @param attributeType the attribute type
     * @param useGenerics   {@code true} if the attribute use a generic type
     * @return attribute type
     */
    public static String getAttributeInterfaceType(ObjectModelParameter attribute,
                                                   String attributeType,
                                                   boolean useGenerics) {
        String result;
        if (attribute instanceof ObjectModelAttribute
            && isNMultiplicity((ObjectModelAttribute) attribute)) {

            result = getCollectionType((ObjectModelAttribute) attribute).getName();
            if (useGenerics) {
                result += "<" + attributeType + ">";
            }
        } else {
            result = attributeType;
        }
        return result;
    }

    public static String getAttributeImplementationType(ObjectModelParameter attribute,
                                                        String attributeType,
                                                        boolean useGenerics) {
        String result;
        if (attribute instanceof ObjectModelAttribute
            && isNMultiplicity((ObjectModelAttribute) attribute)) {

            result = getCollectionInstanceType((ObjectModelAttribute) attribute).getName();
            if (useGenerics) {
                result += "<" + attributeType + ">";
            }
        } else {
            result = attributeType;
        }
        return result;
    }

    public static String getAttributeImplementationType(ObjectModelParameter attribute,
                                                        boolean useGenerics) {
        return getAttributeImplementationType(attribute, attribute.getType(), useGenerics);
    }

    public static String generateName(String prefix, String name, String suffix) {
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotEmpty(prefix)) {
            sb.append(prefix);
        }
        sb.append(name);
        if (StringUtils.isNotEmpty(suffix)) {
            sb.append(suffix);
        }
        return sb.toString();
    }
}
