package org.nuiton.eugene.models.extension.tagvalue;

/*
 * #%L
 * EUGene :: EUGene Core
 * %%
 * Copyright (C) 2004 - 2017 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Joiner;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelPackage;

import java.util.LinkedHashSet;
import java.util.Set;

/**
 * When a missing tag value is detected.
 *
 * Created on 15/08/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 3.0
 */
public class MissingTagValueException extends RuntimeException {

    private static final long serialVersionUID = 1L;

    protected final String tagValueName;

    protected final String tagValueValue;

    protected final String prefixMessage;

    protected final Set<String> stack;

    public MissingTagValueException(String tagValueName, String tagValueValue, String prefixMessage, ObjectModelPackage aPackage, ObjectModelClass aClass) {
        this.tagValueName = tagValueName;
        this.tagValueValue = tagValueValue;
        this.prefixMessage = prefixMessage;
        this.stack = getUsageStack(tagValueName, tagValueValue, aPackage, aClass);
    }

    @Override
    public String toString() {

        return "\n\n" + prefixMessage
                + "\n=========================================================================================="
                + "\n" + Joiner.on("\n").join(stack)
                + "\n==========================================================================================";

    }

    /**
     * Build the stack of usage of the given tag value.
     *
     * Order of usage is : model, packages (from root to final package), then class.
     *
     * @param tagValueName  tag value name
     * @param tagValueValue tag value
     * @param aPackage      package used
     * @param aClass        class used
     * @return the orderer set of stack usage.
     */
    protected LinkedHashSet<String> getUsageStack(String tagValueName, String tagValueValue, ObjectModelPackage aPackage, ObjectModelClass aClass) {
        String suffix = ".tagvalue." + tagValueName + "=" + tagValueValue;
        LinkedHashSet<String> stack = new LinkedHashSet<>();

        stack.add("model" + suffix);

        addPackageTagValue(aPackage, suffix, stack);
        stack.add(aClass.getQualifiedName() + ".class" + suffix);

        return stack;
    }

    protected void addPackageTagValue(ObjectModelPackage aPackage, String suffix, Set<String> stack) {

        if (aPackage.getParentPackage() != null) {
            addPackageTagValue(aPackage.getParentPackage(), suffix, stack);
        }
        stack.add("package." + aPackage.getName() + suffix);

    }

}
