package io.ultreia.java4all.config;

/*-
 * #%L
 * Config :: API
 * %%
 * Copyright (C) 2016 - 2023 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.config.spi.ConfigOptionDef;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import java.util.stream.Collectors;

/**
 * Created on 21/12/2022.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 1.0.25
 */
public class ConfigHelper {

    private static final Logger log = LogManager.getLogger(ConfigHelper.class);

    public static void save(ApplicationConfig applicationConfig, File file, String[] doNotSave, ConfigResource resource, ConfigOptionDef[] options) {
        log.info(String.format("Save configuration to %s", file));
        Properties propertiesToSave = applicationConfig.getPropertiesToSave(false, doNotSave);
        List<String> propertyNames = propertiesToSave.stringPropertyNames().stream().sorted().collect(Collectors.toList());
        for (ConfigOptionDef option : options) {
            String optionKey = option.getKey();
            if (propertyNames.contains(optionKey)) {
                String property = propertiesToSave.getProperty(optionKey);
                if (Objects.equals(property, option.getDefaultValue())) {
                    // use default value
                    propertyNames.remove(optionKey);
                }
            }
        }

        log.info(String.format("Property names to save: %s", propertyNames));
        try (BufferedWriter writer = Files.newBufferedWriter(file.toPath())) {
            List<String> lines = resource.readLines();
            for (String line : lines) {
                if (line.startsWith("#")) {
                    for (String propertyName : propertyNames) {
                        if (line.startsWith("#" + propertyName + "=")) {
                            line = String.format("%s=%s", propertyName, propertiesToSave.getProperty(propertyName));
                        }
                    }
                }
                writer.write(line);
                writer.newLine();
            }
        } catch (IOException e) {
            throw new ApplicationConfigSaveException(e);
        }
    }

}
