package io.ultreia.java4all.config;

/*
 * #%L
 * Config :: API
 * %%
 * Copyright (C) 2016 - 2018 Code Lutin, Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.config.io.spi.ActionModel;
import io.ultreia.java4all.config.io.spi.ConfigModel;
import io.ultreia.java4all.config.io.spi.OptionModel;

import java.util.LinkedList;
import java.util.Locale;
import java.util.ServiceLoader;

/**
 * Provider of a {@link ApplicationConfig}.
 * <p>
 * Each library of application which use {@link ApplicationConfig} should
 * implements this and add the provider available via the
 * {@link ServiceLoader} mecanism.
 * <p>
 * Using such provider offers a nice way to find out what options can be loaded
 * in a application. It also offers a simply way to generate application
 * config report for documentation.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 1.4.8
 */
public interface ApplicationConfigProvider {

    /**
     * Returns the name of the provided application config.
     * <p>
     * This should be the name of the library or application which offers the configuration.
     *
     * @return the name of the provided application config
     */
    String getName();

    /**
     * Returns the localized description of the configuration.
     *
     * @param locale locale used to render description
     * @return the localized description of the configuration
     */
    String getDescription(Locale locale);

    /**
     * Returns all options offered by the configuration.
     *
     * @return all options offered by the configuration
     * @see ConfigOptionDef
     */
    ConfigOptionDef[] getOptions();

    /**
     * Returns all actions offered by the configuration.
     *
     * @return all actions offered by the configuration.
     * @see ConfigActionDef
     */
    ConfigActionDef[] getActions();

    default ConfigModel toModel(Locale locale) {
        ConfigModel configModel = new ConfigModel();
        configModel.setDescription(getDescription(locale));

        LinkedList<OptionModel> optionModels = new LinkedList<>();
        for (ConfigOptionDef configOptionDef : getOptions()) {
            optionModels.add(ConfigOptionDef.of(configOptionDef));
        }
        configModel.setOptions(optionModels);

        LinkedList<ActionModel> actionModels = new LinkedList<>();
        for (ConfigActionDef configActionDef : getActions()) {
            actionModels.add(ConfigActionDef.of(configActionDef));
        }
        configModel.setActions(actionModels);
        return configModel;
    }
}
