package io.ultreia.java4all.application.context;

/*-
 * #%L
 * Application context
 * %%
 * Copyright (C) 2019 - 2020 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.application.context.spi.ApplicationComponentInstantiateStrategy;
import io.ultreia.java4all.util.ServiceLoaders;
import io.ultreia.java4all.util.SingletonSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.Closeable;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * @author Tony Chemit - dev@tchemit.fr
 * @since 1.0.3
 */
public class ApplicationComponentValueSupplier<O> extends SingletonSupplier<O> implements Closeable {

    private static final Logger log = LogManager.getLogger(ApplicationComponentValueSupplier.class);

    private final Class<O> componentType;

    static <O> ApplicationComponentValueSupplier<O> create(Class<O> componentType, boolean requireNotNull, ApplicationComponentInstantiateStrategy instantiateStrategy, Class<?>... dependencies) {
        switch (instantiateStrategy) {
            case CONSTRUCTOR:
                return new ApplicationComponentValueSupplier<>(componentType, requireNotNull, createFromConstructor(componentType, dependencies));
            case SERVICE_LOADER:
                return new ApplicationComponentValueSupplier<O>(componentType, requireNotNull, createFromServiceLoader(componentType)) {
                    @Override
                    public Optional<O> clear() {
                        log.info(String.format("Reload ServiceLoader for: %s", componentType.getName()));
                        ServiceLoaders.reload(componentType);
                        return super.clear();
                    }
                };
            default:
                return null;
        }
    }

    private static <O> Supplier<O> createFromConstructor(Class<O> componentType, Class<?>... dependencies) {
        int dependenciesLength = dependencies.length;
        if (dependenciesLength == 0) {
            return () -> {
                log.info(String.format("Create component value from default constructor: %s", componentType.getName()));
                try {
                    return componentType.newInstance();
                } catch (Exception e) {
                    throw new IllegalArgumentException("Can't instantiate component: " + componentType.getName(), e);
                }
            };
        }

        return () -> {
            Object[] parameters = new Object[dependenciesLength];
            ApplicationContext applicationContext = ApplicationContext.get();
            log.info(String.format("Create component value from constructor: %s with %d dependencies", componentType.getName(), dependenciesLength));
            for (int i = 0; i < dependenciesLength; i++) {
                Class<?> dependency = dependencies[i];
                log.info(String.format(">> Get dependency %3d/%3d - %s", i, dependenciesLength, dependency.getName()));
                ApplicationComponent<?> component = applicationContext.getComponent(dependency);
                log.info(String.format("<< Get dependency %3d/%3d - %s : %s", i, dependenciesLength, dependency.getName(), component));
                parameters[i] = component.get();
            }
            try {
                return componentType.getDeclaredConstructor(dependencies).newInstance(parameters);
            } catch (Exception e) {
                throw new IllegalArgumentException("Can't get component: " + componentType.getName(), e);
            }
        };
    }

    private static <O> Supplier<O> createFromServiceLoader(Class<O> componentType) {
        return () -> {
            try {
                log.info(String.format("Create component value from service loader: %s", componentType.getName()));
                return ServiceLoaders.loadUniqueService(componentType);
            } catch (Exception e) {
                throw new IllegalArgumentException("Can't get component: " + componentType.getName(), e);
            }
        };
    }

    public ApplicationComponentValueSupplier(Class<O> componentType, boolean requireNotNull, Supplier<O> supplier) {
        super(supplier, requireNotNull);
        this.componentType = componentType;
    }

    @Override
    public O get() {
        if (!withValue()) {
            log.info(String.format("Create component value from supplier: %s - %s", componentType.getName(), this));
        }
        return super.get();
    }

    @Override
    public Optional<O> clear() {
        if (withValue()) {
            log.info(String.format("Clear component value: %s", componentType.getName()));
        }
        return super.clear();
    }

    @Override
    public void close() {
        clear();
    }
}
