package io.ultreia.java4all.application.context;

/*-
 * #%L
 * Application context
 * %%
 * Copyright (C) 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.application.context.spi.ApplicationComponentInstantiateStrategy;
import io.ultreia.java4all.application.context.spi.GenerateApplicationComponent;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;


/**
 * Application component is a unique instance shared by {@link ApplicationContext}.
 * <p>
 * To generate a such component, just place the {@link GenerateApplicationComponent} annotation on it.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @see GenerateApplicationComponent
 * @since 1.0.0
 */
@SuppressWarnings("unused")
public class ApplicationComponent<O> implements Supplier<O>, Closeable {

    private static final Logger log = LogManager.getLogger(ApplicationComponent.class);

    /**
     * Name of the component.
     */
    private final String name;
    /**
     * Type of the application component.
     */
    private final Class<O> componentType;
    /**
     * All dependencies (registered also as application components).
     */
    private final List<Class<?>> dependencies;
    /**
     * All hints to get this component via {@link ApplicationContext}.
     */
    private final List<Class<?>> hints;
    /**
     * Delegate singleton supplier to get component value.
     */
    private final ApplicationComponentValueSupplier<O> supplier;

    protected ApplicationComponent(String name, Class<O> componentType, boolean requireNotNull, ApplicationComponentInstantiateStrategy instantiateStrategy, Class<?>[] hints, Class<?>[] dependencies) {
        this(name, componentType, requireNotNull, instantiateStrategy, hints, dependencies, ApplicationComponentValueSupplier.create(componentType, requireNotNull, instantiateStrategy, dependencies));
    }

    protected ApplicationComponent(String name, Class<O> componentType, boolean requireNotNull, ApplicationComponentInstantiateStrategy instantiateStrategy, Class<?>[] hints, Class<?>[] dependencies, ApplicationComponentValueSupplier<O> supplier) {
        this.name = name;
        this.componentType = componentType;
        this.dependencies = new LinkedList<>();
        Collections.addAll(this.dependencies, dependencies);
        this.hints = new LinkedList<>();
        Collections.addAll(this.hints, hints);
        this.supplier = supplier;
    }

    @Override
    public O get() {
        return supplier.get();
    }

    public String getName() {
        return name;
    }

    public Class<O> getComponentType() {
        return componentType;
    }

    public void setValue(O value) {
        supplier.setSupplier(() -> value);
    }

    @Override
    public void close() throws IOException {
        log.info("Closing component: " + this);
        if (supplier.withValue()) {
            if (Closeable.class.isAssignableFrom(componentType)) {
                ((Closeable) supplier.get()).close();
            }
            supplier.clear();
        }
    }

    public boolean withDependencies() {
        return !getDependencies().isEmpty();
    }

    public List<Class<?>> getDependencies() {
        return dependencies;
    }

    public void addExtraHints(Class<?>... hints) {
        Collections.addAll(this.hints, hints);
    }

    public List<Class<?>> getHints() {
        return hints;
    }

    public String toString() {
        return String.format("%s → %s", name, componentType.getName());
    }

}
