package io.ultreia.gc.rest.client;

/*
 * #%L
 * GC toolkit :: REST API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import io.ultreia.gc.http.GcRequest;
import io.ultreia.gc.http.GcRequestBuilder;
import io.ultreia.gc.http.GcResponse;
import io.ultreia.gc.service.api.GcServiceContext;
import io.ultreia.gc.service.internal.GcServiceSupport;
import io.ultreia.gc.session.GcSession;

/**
 * Created on 30/08/15.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public abstract class GcRestClientServiceSupport extends GcServiceSupport {

    private String apiUrl;

    @Override
    public void setServiceContext(GcServiceContext serviceContext) {
        super.setServiceContext(serviceContext);
        GcRestClientConfig config = getApplicationContext().getConfig();
        apiUrl = config.getApiUrl().toExternalForm();
        if (!apiUrl.endsWith("/")) {
            apiUrl += "/";
        }
    }

    protected GcRestClientApplicationContext getApplicationContext() {
        return (GcRestClientApplicationContext) serviceContext;
    }

    @Override
    public GcRequestBuilder forGet(String baseUrl) {
        GcRequestBuilder requestBuilder = GcRequestBuilder.forGet(apiUrl + baseUrl, this::getAuthenticationToken);
        addAuthId(requestBuilder);
        return requestBuilder.setTimeout(60 * 60 * 1000);
    }

    private String getAuthenticationToken() {
        return serviceContext.getAuthId().orElse(null);
    }

    @Override
    public GcRequestBuilder forPost(String baseUrl) {
        GcRequestBuilder requestBuilder = GcRequestBuilder.forPost(apiUrl + baseUrl, this::getAuthenticationToken);
        addAuthId(requestBuilder);
        return requestBuilder.setTimeout(60 * 60 * 1000);
    }

    @Override
    public GcRequestBuilder forDelete(String baseUrl) {
        GcRequestBuilder requestBuilder = GcRequestBuilder.forDelete(apiUrl + baseUrl, this::getAuthenticationToken);
        addAuthId(requestBuilder);
        return requestBuilder.setTimeout(60 * 60 * 1000);
    }

    @Override
    public GcResponse executeRequest(GcRequest request) {
        return getApplicationContext().executeRequest(request);
    }

    @Override
    public GcResponse executeRequest(GcRequest request, int expectedStatusCode) {
        GcResponse gcResponse = getApplicationContext().executeRequest(request);
        GcSession.checkStatusCode(gcResponse, expectedStatusCode);
        return gcResponse;
    }

    private void addAuthId(GcRequestBuilder requestBuilder) {
        if (getApplicationContext().withAuthId()) {
            requestBuilder.addAuthToken();
        }
    }

}
