package io.ultreia.gc.rest.client;

/*
 * #%L
 * GC toolkit :: REST API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;
import io.ultreia.gc.config.GcApiConfig;
import io.ultreia.gc.http.GcExecuteRequestException;
import io.ultreia.gc.http.GcRequest;
import io.ultreia.gc.http.GcResponse;
import io.ultreia.gc.http.GcResponseBuilder;
import io.ultreia.gc.model.GcCache;
import io.ultreia.gc.model.GcLog;
import io.ultreia.gc.service.api.GcCacheService;
import io.ultreia.gc.service.api.GcLogService;
import io.ultreia.gc.service.api.GcService;
import io.ultreia.gc.service.internal.GcCacheServiceImpl;
import io.ultreia.gc.service.internal.GpxServiceImpl;
import io.ultreia.gc.ui.GcApplicationContextSupport;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Tony Chemit - dev@tchemit.fr
 */
public class GcRestClientApplicationContext extends GcApplicationContextSupport<GcRestClientConfig> {

    /** Logger. */
    private static final Log log = LogFactory.getLog(GcRestClientApplicationContext.class);

    private static GcRestClientApplicationContext INSTANCE;

    public static GcRestClientApplicationContext create(GcRestClientConfig config) {
        return INSTANCE = new GcRestClientApplicationContext(config);
    }

    static GcRestClientApplicationContext get() {
        return INSTANCE;
    }

    private final GcResponseBuilder responseBuilder = GcResponseBuilder.create();

    private GcRestClientApplicationContext(GcRestClientConfig config) {
        super(config);
    }

    boolean withAuthId() {
        return getAuthId().isPresent();
    }

    protected <S extends GcService> Class<S> getServiceClass(Class<S> serviceType) {

        if (!serviceType.isInterface() && !Modifier.isAbstract(serviceType.getModifiers())) {
            return serviceType;
        }
        try {
            return (Class) Class.forName(serviceType.getName().replace("service.api", "rest.client.v1") + "RestClient");
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException(e);
        }
    }

    GcResponse executeRequest(GcRequest request) {
        try {
            return responseBuilder.executeRequest(request);
        } catch (IOException e) {
            throw new GcExecuteRequestException(e);
        }
    }

    public static void main(String[] args) throws IOException {

        GcRestClientConfig clientConfig = GcRestClientConfig.create(args);

        GcRestClientApplicationContext applicationContext = GcRestClientApplicationContext.create(clientConfig);

        String authId = applicationContext.newLoginService().login(clientConfig.getLogin(), clientConfig.getPassword());

        log.info("log in: " + authId);

        applicationContext.newLogService().loadLogs();

//        GcApiConfig config = GcApiConfig.create(args);
//        List<GcCache> gcCaches = loadMyFinds(config.getMyFindsGpx(), config.getMyFindsJson());
//        GcChallengeResult check = new PenitentChallengeChecker(gcCaches).check("");
//        log.info(check.getLog());

//        loadLogs(config, cacheService, logService, i);


//
//        caches.parallelStream().forEach(cache -> {
//            log.info(String.format("update cache: %s", cache.getGcName()));
//            try {
//                cacheService.storeCache(cache);
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//        });

    }

    static List<GcCache> loadMyFinds(Path gpxPath, Path myDbPath) throws IOException {

        List<GcCache> caches;
        if (!Files.exists(myDbPath) || Files.getLastModifiedTime(myDbPath).compareTo(Files.getLastModifiedTime(gpxPath)) < 0) {
            log.info("Loading gpx " + gpxPath);
            caches = new GpxServiceImpl().getCaches(gpxPath.toFile());
            log.info("Load " + caches.size() + " cache(s).");

            new GcCacheServiceImpl().storeCaches(caches, myDbPath.toFile());
        } else {

            log.info("Loading json " + myDbPath);
            caches = new GcCacheServiceImpl().loadCaches(myDbPath.toFile());
            log.info("Load " + caches.size() + " cache(s).");
        }
        return caches;
    }

    static void loadLogs(GcApiConfig config, GcRestClientApplicationContext applicationContext) throws IOException {

        MutableInt i = new MutableInt();

        GcCacheService cacheService = applicationContext.newCacheService();
        GcLogService logService = applicationContext.newLogService();

        List<String> gcNames;

        Path path = config.getDataDirectory().toPath().resolve("gcnames.json");
        if (Files.exists(path)) {
            try (BufferedReader writer = Files.newBufferedReader(path)) {
                Type type = new TypeToken<List<String>>() {
                }.getType();
                gcNames = new GsonBuilder().setPrettyPrinting().create().fromJson(writer, type);
            }
        } else {
            gcNames = new LinkedList<>(cacheService.getGcNames());
            try (BufferedWriter writer = Files.newBufferedWriter(path)) {
                new GsonBuilder().setPrettyPrinting().create().toJson(gcNames, writer);
            }
        }
        System.setProperty("java.util.concurrent.ForkJoinPool.common.parallelism", "20");

        gcNames.sort(String::compareTo);
        Collections.reverse(gcNames);
        gcNames.parallelStream().forEach(gcName -> {

            try {
                log.info(String.format("%d/%d : %s", i.incrementAndGet(), gcNames.size(), gcName));
                GcCache cache = cacheService.getCacheFromGcName(gcName);
                List<GcLog> logsFromGcName = logService.getLogsFromGcName(gcName);
                log.info(String.format("%d/%d : %s - %d", i.getValue(), gcNames.size(), cache.getGcName(), logsFromGcName.size()));
            } catch (Exception e) {
                e.printStackTrace();
            }
        });
    }
}
