package io.ultreia.gc.service;

/*-
 * #%L
 * GC toolkit :: API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.github.mustachejava.MustacheFactory;
import io.ultreia.gc.config.GcApiConfig;
import io.ultreia.gc.http.GcRequest;
import io.ultreia.gc.http.GcRequestBuilder;
import io.ultreia.gc.http.GcResponse;
import io.ultreia.gc.model.GcCacheImage;
import io.ultreia.gc.model.GcFieldNote;
import io.ultreia.gc.model.GcLogDraft;
import io.ultreia.gc.model.GcTrackable;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.HttpStatus;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.nuiton.config.ArgumentsParserException;

/**
 * Created by tchemit on 16/04/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class FieldNotesService extends GcServiceSupport {

    /** Logger. */
    private static final Log log = LogFactory.getLog(FieldNotesService.class);

    private static final String URL_GET_FIELD_NOTES = "https://www.geocaching.com/my/fieldnotes.aspx";
    private static final String URL_GET_FIELD_NOTE_FORM = "https://www.geocaching.com/my/%s";
    private static final String URL_SEND_LOG_IMAGE = "https://www.geocaching.com/api/proxy/web/v1/LogDrafts/images";
    private static final String URL_SEND_LOG = "https://www.geocaching.com/api/proxy/web/v1/Geocache/%s/GeocacheLog";
    private static final String URL_ATTACH_IMAGE_TO_LOG = "https://www.geocaching.com/api/proxy/web/v1/LogDrafts/images/%s?geocacheLogReferenceCode=%s";
    private static final String URL_DELETE_LOG_DRAFT = "https://www.geocaching.com/api/proxy/web/v1/LogDrafts/%s";
    private static final String URL_GET_LOG_DRAFT = "https://www.geocaching.com/api/proxy/web/v1/LogDrafts/%s";

    private MustacheFactory mustacheFactory;

    /**
     * @return the names of caches you got in your Geocaching.com field notes
     */
    public List<GcFieldNote> loadFieldNotes() {

        GcRequest request = forGet(URL_GET_FIELD_NOTES).build();

        GcResponse build = executeRequest(request, HttpStatus.SC_OK);

        Document doc = build.getResponseAsHtml();

        Elements trs = doc.select("table[class=Table] > tbody > tr");

        List<GcFieldNote> fieldNotes = new LinkedList<>();
        for (Element tr : trs) {

            Elements select = tr.select("td > a[href^=https://www]");
            String name = select.text();
            String url = select.attr("href");

            String date = tr.child(2).text();
            String logType = tr.child(3).text();

            String composeUrl = tr.child(4).select(" a[href^=fieldnotes.aspx?composeLog=]").first().attr("href");

            fieldNotes.add(new GcFieldNote(name, url, composeUrl, date, logType));

        }
        log.info("Found " + fieldNotes.size() + " field note(s).");
        fieldNotes.forEach(c -> log.debug("Cache " + c + " - " + c.getComposeUrl()));
        return fieldNotes;
    }


    public void logFieldNote(GcFieldNote fieldNote, String logText, boolean autoVisitTrackables, ProgressMonitor progressBarModel) {

        progressBarModel.increment("Prepare log for cache: " + fieldNote.getName());

        Document doc;
        {
            GcRequest request = forGet(String.format(URL_GET_FIELD_NOTE_FORM, fieldNote.getComposeUrl())).build();

            GcResponse response = executeRequest(request, HttpStatus.SC_OK);
            doc = response.getResponseAsHtml();
        }

        String logId = doc.select("meta[name='og:url']").attr("content").split("=")[1];
        String gcName = doc.select("h1[class='muted'] > a").attr("href").split("/")[2];
        String date = doc.select("input[id='LogDate']").attr("value");
        String ownerName = doc.select("h1[class='muted'] > a:nth-child(2)").text();
        int cacheCount = Integer.valueOf(doc.select("span[class='cache-count']").text().split(" ")[0].replace(",", ""));

        log.debug("LogId: " + logId);
        log.debug("GcName: " + gcName);
        log.debug("cache count: " + cacheCount);
        log.debug("the owner: " + ownerName);

        String logType = null;
        String logTypeName = fieldNote.getLogType().toLowerCase();
        for (Element logTypeOption : doc.select("select[class='log-type'] option")) {
            if (logTypeName.contains(logTypeOption.text().trim().toLowerCase())) {
                logType = logTypeOption.attr("value");
                break;
            }
        }
        Objects.requireNonNull(logType);

        String imageId = null;
        if (fieldNote.getImage().isPresent()) {

            progressBarModel.increment("Upload image for cache: " + fieldNote.getName());

            GcLogDraft logDraft = getLogDraft(logId);
            GcCacheImage gcCacheImage = fieldNote.getImage().get();

            GcRequest request = forPost(URL_SEND_LOG_IMAGE)
                    .addAuthToken()
                    .useMultiPartForm()
                    .addParameter("guid", logDraft.getGuid())
                    .addParameter("name", gcCacheImage.getContent().getName() + System.nanoTime())
                    .addParameter("qqtotalfilesize", gcCacheImage.getContent().length() + "")
                    .addFile("qqfile", gcCacheImage.getContent())
                    .build();

            progressBarModel.increment("Send log for cache: " + fieldNote.getName());

            GcResponse response = executeRequest(request, HttpStatus.SC_CREATED);

            Document responseAsHtml = response.getResponseAsHtml();

            imageId = responseAsHtml.select("guid").text();

        }

        GcRequestBuilder requestBuilder = forPost(String.format(URL_SEND_LOG, gcName))
                .addAuthToken()
                .addHiddenInputs(doc)
                .addParameter("geocache[referenceCode]", gcName)
                .addParameter("logType", logType)
                .addParameter("logDate", date)
                .addParameter("referenceCode", logId)
                .addParameter("logText", getLogText(logText, ownerName, cacheCount))
                .addParameter("addedFavorite", fieldNote.isFavorite());

        if (autoVisitTrackables) {
            for (GcTrackable trackable : getTrackables()) {
                requestBuilder.addParameter("trackableActivity[" + trackable.getReferenceCode() + "]", "75");
            }
        }

        String realLogId;

        {
            progressBarModel.increment("Send log for cache: " + fieldNote.getName());

            GcRequest request = requestBuilder.build();
            GcResponse response = executeRequest(request, HttpStatus.SC_OK);

            realLogId = response.getResponseAsHtml().select("geocachelog > ReferenceCode").text();
            log.info("Real log id: " + realLogId);
        }

        Objects.requireNonNull(realLogId);

        if (imageId != null) {

            progressBarModel.increment("Attach image to log for cache: " + fieldNote.getName());

            GcRequest request = forPatch(String.format(URL_ATTACH_IMAGE_TO_LOG, imageId, realLogId))
                    .addAuthToken().build();
            executeRequest(request, HttpStatus.SC_OK);
        }


        {
            progressBarModel.increment("Delete field note for cache: " + fieldNote.getName());

            GcRequest request = forDelete(String.format(URL_DELETE_LOG_DRAFT, logId)).addAuthToken().build();
            executeRequest(request, HttpStatus.SC_OK);
        }

    }

    private GcLogDraft getLogDraft(String referenceCode) {

        GcRequest request = forGet(String.format(URL_GET_LOG_DRAFT, referenceCode))
                .addAuthToken()
                .build();

        GcResponse response = executeRequest(request, HttpStatus.SC_OK);
        Document responseAsHtml = response.getResponseAsHtml();
        return GcLogDraft.fromXml(responseAsHtml);
    }

    private String getLogText(String logText, String theOwner, int cacheCount) {
        Mustache mustache = getMustacheFactory().compile(new StringReader(logText), "archeo");

        Map<String, Object> mustacheModel = new HashMap<>();
        mustacheModel.put("found", cacheCount + 1);
        mustacheModel.put("owner", theOwner);

        try (StringWriter logTextWriter = new StringWriter()) {
            mustache.execute(logTextWriter, mustacheModel);
            logTextWriter.flush();
            return logTextWriter.toString();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private MustacheFactory getMustacheFactory() {
        if (mustacheFactory == null) {
            mustacheFactory = new DefaultMustacheFactory();
        }
        return mustacheFactory;
    }

    public static void main(String[] args) throws ArgumentsParserException {

        GcApiConfig config = new GcApiConfig();
        config.get().parse(args);

        String sessionId = new GcAuthService().login(config.getLogin(), config.getPassword());
        log.info("SessionId: " + sessionId);

        GcServiceContext serviceContext = GcServiceContext.create(sessionId);

        FieldNotesService service2 = new FieldNotesService();
        service2.setServiceContext(serviceContext);

        List<GcFieldNote> gcFieldNotes = service2.loadFieldNotes();
        GcFieldNote gcFieldNote = gcFieldNotes.get(0);

        GcCacheImage image = new GcCacheImage();
        image.setContent(config.getDataDirectory().toPath().resolve("taiste.jpg").toFile());
        image.setTitle("FTF!");
        gcFieldNote.setImage(image);
        service2.logFieldNote(gcFieldNote, "{{found}} - Ftf log à zuivre.... MPLC {{owner}}", true, new ProgressMonitor() {
            @Override
            public void increment(String stepMessage) {

            }

            @Override
            public void setValue(int size) {

            }
        });


    }
}
