package io.ultreia.gc.http;

/*-
 * #%L
 * GC toolkit :: API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.gson.GsonBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpRequestBase;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;

/**
 * Created by tchemit on 14/04/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class GcResponse implements Closeable {

    private final HttpRequestBase httpRequestBase;
    private final HttpResponse response;
    private final Integer statusCode;

    private final String responseAsString;

    private final Header[] responseHeaders;

    GcResponse(HttpRequestBase httpRequestBase, HttpResponse response, Integer statusCode, String responseAsString, Header... responseHeaders) {
        this.httpRequestBase = httpRequestBase;
        this.response = response;
        this.statusCode = statusCode;
        this.responseAsString = responseAsString;
        this.responseHeaders = responseHeaders;
    }

    public HttpResponse getResponse() {
        return response;
    }

    public Integer getStatusCode() {
        return statusCode;
    }

    public String getResponseAsString() {
        return responseAsString;
    }

    public Header[] getResponseHeaders() {
        return responseHeaders;
    }

    @Override
    public void close() throws IOException {

        try {
            if (response != null && response.getEntity() != null) {
                response.getEntity().getContent().close();
            }
        } finally {

            // Release the connection.
            httpRequestBase.releaseConnection();

        }

    }

    public Document getResponseAsHtml() {
        return Jsoup.parse(responseAsString);
    }

    public Map<String, Object> getResponseAsJson() {
        return new GsonBuilder().setPrettyPrinting().create().fromJson(responseAsString, HashMap.class);
    }

    public <T> T getResponseAsJson(Type type) {
        return new GsonBuilder().setPrettyPrinting().create().fromJson(responseAsString, type);
    }
}
