package io.ultreia.gc.http;

/*
 * #%L
 * GC toolkit :: API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.message.BasicNameValuePair;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

/**
 * @author Tony Chemit - dev@tchemit.fr
 */
public class GcRequestBuilder {

    private final List<NameValuePair> parameters;

    private final Map<String, String> headers;

    protected final Map<String, File> files;

    private String requestBody;

    private String contentType;

    private String requestMethod;

    private int timeout = 30 * 1000; // en millisecondes
    private boolean addAuthToken;
    private boolean useMultipartForm;
    private final String baseUrl;
    private final Supplier<String> authTokenSupplier;

    public static GcRequestBuilder forGet(String baseUrl, Supplier<String> authTokenSupplier) {
        return new GcRequestBuilder(HttpGet.METHOD_NAME, baseUrl, authTokenSupplier);
    }

    public static GcRequestBuilder forPost(String baseUrl, Supplier<String> authTokenSupplier) {
        return new GcRequestBuilder(HttpPost.METHOD_NAME, baseUrl, authTokenSupplier);
    }

    public static GcRequestBuilder forPut(String baseUrl, Supplier<String> authTokenSupplier) {
        return new GcRequestBuilder(HttpPut.METHOD_NAME, baseUrl, authTokenSupplier);
    }

    public static GcRequestBuilder forDelete(String baseUrl, Supplier<String> authTokenSupplier) {
        return new GcRequestBuilder(HttpDelete.METHOD_NAME, baseUrl, authTokenSupplier);
    }

    public static GcRequestBuilder forPatch(String baseUrl, Supplier<String> authTokenSupplier) {
        return new GcRequestBuilder(HttpPatch.METHOD_NAME, baseUrl, authTokenSupplier);
    }

    public GcRequest build() {

        if (addAuthToken) {
            addHeader("Authorization", authTokenSupplier.get());
        }
        Objects.requireNonNull(baseUrl, "'baseUrl' can't be null");
        Objects.requireNonNull(requestMethod, "'requestMethod' was not setted");

        return new GcRequest(
                requestMethod,
                baseUrl,
                contentType,
                requestBody,
                headers,
                parameters,
                files,
                timeout,
                useMultipartForm);
    }

    public GcRequestBuilder addAuthToken() {
        this.addAuthToken = true;
        return this;
    }

    public GcRequestBuilder useMultiPartForm() {
        this.useMultipartForm = true;
        return this;
    }

    public GcRequestBuilder setTimeout(int timeout) {
        this.timeout = timeout;
        return this;
    }

    public GcRequestBuilder setRequestBody(String requestBody) {
        this.requestBody = requestBody;
        return this;
    }

    public GcRequestBuilder setContentType(String contentType) {
        this.contentType = contentType;
        return this;
    }

    public GcRequestBuilder addHeader(String key, String value) {
        checkRequestNotNull(key, value);
        headers.put(key, value);
        return this;
    }

    public GcRequestBuilder addParameter(String parameterName, String value) {
        checkParameterNotNull(parameterName, value);
        parameters.add(new BasicNameValuePair(parameterName, value));
        return this;
    }

    public GcRequestBuilder addParameter(String parameterName, boolean value) {
        checkParameterNotNull(parameterName, value);
        parameters.add(new BasicNameValuePair(parameterName, String.valueOf(value)));
        return this;
    }

    public GcRequestBuilder addFile(String value, File file) {
        checkRequestNotNull(value, file);
        files.put(value, file);
        return this;
    }

    public GcRequestBuilder addHiddenInputs(Document doc) {

        for (Element element : doc.select("input[type='hidden']")) {

            String value = element.attr("value");
            if (value != null) {
                addParameter(element.attr("name"), value);
            }
        }
        return this;
    }

    private GcRequestBuilder(String methodName, String baseUrl, Supplier<String> authTokenSupplier) {
        this.authTokenSupplier = authTokenSupplier;
        Objects.requireNonNull(baseUrl);
        this.baseUrl = baseUrl;
        Objects.requireNonNull(methodName);
        this.requestMethod = methodName;
        this.headers = new LinkedHashMap<>();
        this.files = new LinkedHashMap<>();
        this.parameters = new LinkedList<>();
    }

    private void checkParameterNotNull(String parparameterNamemName, Object value) {
        checkKeyValueNotNull(parparameterNamemName, value, "Parameter key must be not null", "Parameter value must be not null for paramName : " + parparameterNamemName);
    }

    private void checkRequestNotNull(String key, Object value) {
        checkKeyValueNotNull(key, value, "Request key must be not null", "Request value must be not null for key : " + key);
    }

    private void checkKeyValueNotNull(String key, Object value, String keyErrorMessage, String valueErrorMessage) {
        Objects.requireNonNull(key, keyErrorMessage);
        Objects.requireNonNull(value, valueErrorMessage);
    }

    public GcRequestBuilder addHeaders(Map<String, String> headers) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            addHeader(entry.getKey(), entry.getValue());
        }
        return this;
    }
}
