package io.ultreia.gc.challenge;

/*-
 * #%L
 * GC toolkit :: API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import io.ultreia.gc.config.GcApiConfig;
import io.ultreia.gc.model.GcCache;
import io.ultreia.gc.service.GcCacheService;
import io.ultreia.gc.service.GpxService;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.config.ArgumentsParserException;

/**
 * Created by tchemit on 19/04/17.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class Lavoir90Challenge {

    /** Logger. */
    private static final Log log = LogFactory.getLog(Lavoir90Challenge.class);

    public static void main(String[] args) throws ArgumentsParserException {

        GcApiConfig config = new GcApiConfig();
        config.get().parse(args);

        Path gpxPath = config.getMyFindsGpx();
        Path myDbPath = config.getMyFindsJson();

        List<GcCache> caches;
        if (Files.exists(myDbPath)) {

            log.info("Loading " + myDbPath);
            caches = new GcCacheService().loadCaches(myDbPath.toFile());
        } else if (Files.exists(gpxPath)) {
            log.info("Loading " + gpxPath);
            caches = new GpxService().getCaches(gpxPath.toFile());

        } else {
            throw new IllegalStateException("Nothing found to load.");
        }

        log.info("Load " + caches.size() + " cache(s).");

        List<GcCache> lavoirsCaches = caches.stream().filter(c -> c.getDesc().toLowerCase().contains("lavoir")).collect(Collectors.toList());

        Set<String> types = lavoirsCaches.stream().map(GcCache::getGcType).collect(Collectors.toSet());

        double score = lavoirsCaches.stream().mapToDouble(c -> c.getDifficulty() + c.getTerrain()).sum();

        StringBuilder result = new StringBuilder("\n");
        result.append("# Summary #\n\n");
        result.append(String.format("\nFound %d «Lavoir» caches.", lavoirsCaches.size()));
        result.append(String.format("\nFound %d different types of «Lavoir» caches : %s", types.size(), types));
        result.append(String.format("\nYour score is %s.", score));
        result.append("\n\n# Details #\n\n");
        float totalScore = 0;
        for (GcCache lavoir : lavoirsCaches) {
            float cacheScore = lavoir.getDifficulty() + lavoir.getTerrain();
            totalScore += cacheScore;
            result.append(String.format("\n * [%s - %s - %s points] «%s» = %s", lavoir.getGcType(), lavoir.getGcName(), totalScore, lavoir.getDesc(), cacheScore));
        }
        log.info(result.toString());
    }

}
