package io.ultreia.gc.http;

/*
 * #%L
 * GC toolkit :: API
 * %%
 * Copyright (C) 2017 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.File;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.message.BasicNameValuePair;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

/**
 * @author Tony Chemit - dev@tchemit.fr
 */
public class GcRequestBuilder {

    protected final List<NameValuePair> parameters;

    protected final Map<String, String> headers;

    protected final Map<String, File> files;

    protected String requestBody;

    protected String contentType;

    protected String requestMethod;

    protected int timeout = 30 * 1000; // en millisecondes

    public static GcRequestBuilder forGet() {
        return new GcRequestBuilder(HttpGet.METHOD_NAME);
    }

    public static GcRequestBuilder forPost() {
        return new GcRequestBuilder(HttpPost.METHOD_NAME);
    }

    public static GcRequestBuilder forPut() {
        return new GcRequestBuilder(HttpPut.METHOD_NAME);
    }

    public static GcRequestBuilder forDelete() {
        return new GcRequestBuilder(HttpDelete.METHOD_NAME);
    }

    public GcRequest build(String baseUrl) {

        Objects.requireNonNull(baseUrl, "'baseUrl' can't be null");
        Objects.requireNonNull(requestMethod, "'requestMethod' was not setted");

        return new GcRequest(
                requestMethod,
                baseUrl,
                contentType,
                requestBody,
                headers,
                parameters,
                files,
                timeout
        );
    }

    public GcRequestBuilder setTimeout(int timeout) {
        this.timeout = timeout;
        return this;
    }

    public GcRequestBuilder setRequestBody(String requestBody) {
        this.requestBody = requestBody;
        return this;
    }

    public GcRequestBuilder setContentType(String contentType) {
        this.contentType = contentType;
        return this;
    }

    public GcRequestBuilder addHeader(String key, String value) {
        checkRequestNotNull(key, value);
        headers.put(key, value);
        return this;
    }

    public GcRequestBuilder addParameter(String parameterName, String value) {
        checkParameterNotNull(parameterName, value);
        parameters.add(new BasicNameValuePair(parameterName, value));
        return this;
    }

    public GcRequestBuilder addHiddenInputs(Document doc) {

        for (Element element : doc.select("input[type='hidden']")) {

            String value = element.attr("value");
            if (value != null) {
                addParameter(element.attr("name"), value);
            }
        }
        return this;
    }

    private GcRequestBuilder(String methodName) {
        Objects.requireNonNull(methodName);
        this.requestMethod = methodName;
        this.headers = new LinkedHashMap<>();
        this.files = new LinkedHashMap<>();
        this.parameters = new LinkedList<>();
    }

    protected void checkParameterNotNull(String parparameterNamemName, Object value) {
        checkKeyValueNotNull(parparameterNamemName, value, "Parameter key must be not null", "Parameter value must be not null for paramName : " + parparameterNamemName);
    }

    protected void checkRequestNotNull(String key, Object value) {
        checkKeyValueNotNull(key, value, "Request key must be not null", "Request value must be not null for key : " + key);
    }

    protected void checkKeyValueNotNull(String key, Object value, String keyErrorMessage, String valueErrorMessage) {
        Objects.requireNonNull(key, keyErrorMessage);
        Objects.requireNonNull(value, valueErrorMessage);
    }

    public GcRequestBuilder addCookie(String cookieName, String value) {
        return this;
    }

    public GcRequestBuilder addHeaders(Map<String, String> headers) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            addHeader(entry.getKey(), entry.getValue());
        }
        return this;
    }
}
