/*
 * Copyright 2021 Red Date Technology Ltd.  Licensed under MPLv2
 * (https://www.mozilla.org/en-US/MPL/2.0/)
 */

package io.udpn.commonsshutdownbootstarter.endpoint;

import io.udpn.commonsshutdownbootstarter.support.ShutdownMarker;
import io.udpn.commonsshutdownbootstarter.support.ContextShutdown;
import io.udpn.commonsshutdownbootstarter.event.RedShutdownEvent;
import java.util.Collections;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.actuate.endpoint.annotation.WriteOperation;
import org.springframework.boot.actuate.health.Status;
import org.springframework.cloud.client.serviceregistry.Registration;
import org.springframework.cloud.client.serviceregistry.ServiceRegistry;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

/**
 * {@link Endpoint @Endpoint} to shutdown the {@link ApplicationContext}.
 *
 * @author wb
 * @version 1.0.0
 * @Date: 2024/1/25
 * @history date, modifier,and description
 **/
@Endpoint(id = RedShutdownEndpoint.ENDPOINT)
public class RedShutdownEndpoint implements ApplicationContextAware {

  public static final String ENDPOINT = "redshutdown";

  private static final Logger log = LoggerFactory.getLogger(RedShutdownEndpoint.class);

  private static final Map<String, String> SHUTDOWN_MESSAGE = Collections.singletonMap("message",
      "Shutting down, bye...");

  private ApplicationContext context;

  private final Registration registration;

  private final ServiceRegistry serviceRegistry;

  public RedShutdownEndpoint(ServiceRegistry<?> serviceRegistry, Registration registration) {
    this.serviceRegistry = serviceRegistry;
    this.registration = registration;
  }

  @WriteOperation
  public Map<String, String> safeShutdown() {
    if (ShutdownMarker.isShutdown()) {
      return SHUTDOWN_MESSAGE;
    }
    // Instance offline
    instanceOffline();

    // Publish shutdown event.
    publishShutdownEvent();

    // Context close
    ContextShutdown.safeShutdown();
    return SHUTDOWN_MESSAGE;
  }

  private void instanceOffline() {
    if (this.registration == null || this.serviceRegistry == null) {
      log.warn("No registration found. Skip...");
    } else {
      this.serviceRegistry.setStatus(this.registration, Status.DOWN.getCode());
    }
  }

  private void publishShutdownEvent() {
    try {
      // Publish shutdown event.
      context.publishEvent(new RedShutdownEvent(this));
    } catch (Throwable ex) {
      log.warn("Exception thrown from ApplicationListener handling RedShutdownEvent", ex);
    }
  }

  @Override
  public void setApplicationContext(ApplicationContext context) throws BeansException {
    this.context = context;
  }

}
