/*
 * Copyright 2021 Red Date Technology Ltd.  Licensed under MPLv2
 * (https://www.mozilla.org/en-US/MPL/2.0/)
 */

package io.udpn.commonsshutdownbootstarter.configuration;

import io.udpn.commonsshutdownbootstarter.endpoint.RedShutdownEndpoint;
import io.udpn.commonsshutdownbootstarter.taskshutdown.TaskRegistryEndpoint;
import java.time.Duration;
import java.util.Arrays;
import java.util.LinkedHashMap;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.context.config.ConfigFileApplicationListener;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.boot.web.server.Shutdown;
import org.springframework.boot.web.server.WebServer;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MapPropertySource;

/**
 * Set the {@link WebServer} to graceful shutdown by default
 *
 * @author wb
 * @version 1.0.0
 * @Date: 2024/1/26
 * @history date, modifier,and description
 **/
public class ShutdownEnvironmentPostProcessor implements EnvironmentPostProcessor, Ordered {

  // After ConfigFileApplicationListener
  private int order = ConfigFileApplicationListener.DEFAULT_ORDER + 1;

  public static final String ENVIRONMENT_NAME = "redshutdown";

  @Override
  public void postProcessEnvironment(ConfigurableEnvironment environment,
      SpringApplication application) {
    LinkedHashMap<String, Object> map = new LinkedHashMap<>();
    map.put("server.shutdown", Shutdown.GRACEFUL);
    map.put("spring.lifecycle.timeout-per-shutdown-phase", Duration.ofSeconds(120L));
    setDefaultShutdownEndpoint(map);
    MapPropertySource propertySource = new MapPropertySource(
        ENVIRONMENT_NAME, map);
    environment.getPropertySources().addLast(propertySource);
  }

  private void setDefaultShutdownEndpoint(LinkedHashMap<String, Object> map) {
    // {@link RedShutdownEndpoint} added to the default[DefaultIncludes.WEB] endpoint
    map.put("management.endpoints.web.exposure.include",
        Arrays.asList("info", "health", RedShutdownEndpoint.ENDPOINT, TaskRegistryEndpoint.ENDPOINT));
  }

  @Override
  public int getOrder() {
    return order;
  }
}
