/*
 * MIT License
 *
 * Copyright (c) 2022. Toxicity
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 **/
package io.toxicity.rsa.api.key.validator

import io.matthewnelson.component.base64.decodeBase64ToArray
import io.matthewnelson.component.encoding.base32.Base32
import io.toxicity.rsa.api.key.validator.ext.isExpiredInternal
import io.toxicity.rsa.api.key.validator.ext.verifyRsaSignature
import kotlin.jvm.JvmField
import kotlin.jvm.JvmStatic

class RsaApiKeyMessage private constructor(
    @JvmField
    val isDevKey: Boolean,
    @JvmField
    val packageName: String,
    @JvmField
    val type: Type,
    @JvmField
    val expiration: String,
    @JvmField
    val uuid: String
) {

    enum class Type {
        PUSH,
        MESSAGE
    }

    fun isExpired(): Boolean {
        return expiration.isExpiredInternal()
    }

    override fun equals(other: Any?): Boolean {
        return  other is RsaApiKeyMessage           &&
                other.isDevKey == isDevKey          &&
                other.packageName == packageName    &&
                other.type == type                  &&
                other.expiration == expiration      &&
                other.uuid == uuid
    }

    override fun hashCode(): Int {
        var result = 17
        result = result * 31 + isDevKey.hashCode()
        result = result * 31 + packageName.hashCode()
        result = result * 31 + type.hashCode()
        result = result * 31 + expiration.hashCode()
        result = result * 31 + uuid.hashCode()
        return result
    }

    override fun toString(): String = """
        RsaApiKeyMessage(
            $ARG_IS_DEV_KEY$isDevKey,
            $ARG_PACKAGE_NAME$packageName,
            $ARG_TYPE$type,
            $ARG_EXPIRATION$expiration,
            $ARG_UUID$uuid
        )
    """.trimIndent()

    companion object {
        const val ARG_IS_DEV_KEY = "isDevKey="
        const val ARG_PACKAGE_NAME = "packageName="
        const val ARG_TYPE = "type="
        const val ARG_EXPIRATION = "expiration="
        const val ARG_UUID = "uuid="

        const val EXPIRATION_NEVER = "NEVER"

        const val UTC = "UTC"
        const val DATE_FORMAT = "yyyy-MM-dd' 'HH:mm:ss' $UTC'"

        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun fromString(message: String): RsaApiKeyMessage {
            val lines = message.lines()

            val isDevKey: Boolean = lines
                .elementAtOrNull(0)
                ?.let {
                    it.substringAfter(ARG_IS_DEV_KEY)
                        .toBooleanStrictOrNull()
                        ?: throw IllegalArgumentException(
                            "Failed to determine $ARG_IS_DEV_KEY value from $it"
                        )
                }
                ?: throw IllegalArgumentException("Failed to determine $ARG_IS_DEV_KEY value")

            val packageName: String = lines
                .elementAtOrNull(1)
                ?.let {
                    val name = it.substringAfter(ARG_PACKAGE_NAME)
                    require(name.isNotEmpty()) {
                        "Package name cannot be empty"
                    }
                    name
                }
                ?: throw IllegalArgumentException("Failed to determine $ARG_PACKAGE_NAME value")

            val type: Type = lines
                .elementAtOrNull(2)
                ?.substringAfter(ARG_TYPE)
                ?.let {
                    try {
                        Type.valueOf(it)
                    } catch (e: Exception) {
                        throw IllegalArgumentException("" +
                                "Failed to determine $ARG_TYPE value from $it", e
                        )
                    }
                } ?: throw IllegalArgumentException("Failed to determine $ARG_TYPE value")

            val expiration: String = lines
                .elementAtOrNull(3)
                ?.substringAfter(ARG_EXPIRATION)
                ?.let { dateString ->
                    if (dateString == EXPIRATION_NEVER) {
                        dateString
                    } else {
                        try {
                            dateString.isExpiredInternal()

                            // did not throw so it is a valid format date string
                            dateString
                        } catch (e: Exception) {
                            throw IllegalArgumentException(
                                "Failed to determine $ARG_EXPIRATION from $dateString", e
                            )
                        }
                    }
                }
                ?: throw IllegalArgumentException("Failed to determine $ARG_EXPIRATION value")

            val uuid: String = lines
                .elementAtOrNull(4)
                ?.substringAfter(ARG_UUID)
                ?.let { uuidString ->
                    val exMessage = "Failed to determine $ARG_UUID from $uuidString"

                    require(uuidString.length == 29) {
                        exMessage
                    }

                    require(
                        uuidString
                            .replace("-", "")
                            .matches("[${Base32.Crockford.CHARS}]{24}".toRegex())
                    ) {
                        exMessage
                    }

                    uuidString
                }
                ?: throw IllegalArgumentException("Failed to determine $ARG_UUID value")

            return RsaApiKeyMessage(
                isDevKey = isDevKey,
                packageName = packageName,
                type = type,
                expiration = expiration,
                uuid = uuid,
            )
        }
    }

    object Validator {
        const val ALGO_SIGNATURE = "SHA256withRSA"
        const val ALGO_KEYS = "RSA"
        const val DELIMITER = ':'

        /**
         * Validates an api key by checking the RSA signature.
         *
         * Passing `null` for [b64DevRsaPubKey] will cause a [DevKeyException] to
         * be thrown when an issued Developer RsaApiKey is attempted to be used.
         * */
        @JvmStatic
        @Throws(IllegalArgumentException::class)
        fun validate(
            rsaApiKey: String,
            b64ProdRsaPubKey: String,
            b64DevRsaPubKey: String?
        ): RsaApiKeyMessage {
            val splits = rsaApiKey.split(DELIMITER)
            if (splits.size != 2) {
                throw IllegalArgumentException("Failed to split provided rsaApiKey into message and signature")
            }

            val message = splits[0].decodeBase64ToArray()
                ?: throw IllegalArgumentException("Failed to base64 decode message")

            val apiKeyMessage = fromString(message.decodeToString())

            val pubKey = if (apiKeyMessage.isDevKey) {
                b64DevRsaPubKey ?: throw DevKeyException(
                    "Development Keys do not work in production. A production key must be used."
                )
            } else {
                b64ProdRsaPubKey
            }

            if (!verifySignature(message, pubKey, splits[1])) {
                throw IllegalArgumentException("Failed to verify message's signature")
            }

            return apiKeyMessage
        }

        @Throws(IllegalArgumentException::class)
        private fun verifySignature(
            message: ByteArray,
            b64RsaPubKey: String,
            b64MessageSignature: String,
        ): Boolean {
            val sigBytes = b64MessageSignature.decodeBase64ToArray()
                ?: throw IllegalArgumentException("Failed to decode signature")

            val pubBytes = b64RsaPubKey.decodeBase64ToArray()
                ?: throw IllegalArgumentException("Failed to decode rsa public key")

            return try {
                message.verifyRsaSignature(signature = sigBytes, pubKey = pubBytes)
            } catch (e: Exception) {
                throw IllegalArgumentException("Failed to verify message signature", e)
            }
        }
    }
}
