/*
 * Copyright (c) 2023 Toxicity
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
@file:Suppress("KotlinRedundantDiagnosticSuppress")

package io.toxicity.apikey

import io.matthewnelson.encoding.base32.Base32Crockford
import io.matthewnelson.encoding.base64.Base64
import io.matthewnelson.encoding.core.Decoder.Companion.decodeToByteArray
import io.matthewnelson.encoding.core.Decoder.Companion.decodeToByteArrayOrNull
import io.matthewnelson.encoding.core.Encoder.Companion.encodeToString
import io.matthewnelson.encoding.core.EncodingException
import io.matthewnelson.encoding.core.use
import io.matthewnelson.encoding.core.util.LineBreakOutFeed
import io.toxicity.apikey.ApiKey.Data.Companion.toApiKeyData
import io.toxicity.crypto.GeneralSecurityException
import io.toxicity.crypto.rsa.RSAScope
import io.toxicity.crypto.rsa.RSASignature
import kotlinx.datetime.Clock
import kotlinx.datetime.Instant
import kotlinx.datetime.toInstant
import org.kotlincrypto.hash.sha3.SHA3_224
import kotlin.jvm.*
import kotlin.time.Duration
import kotlin.time.Duration.Companion.days

/**
 * An [ApiKey] consists of 2 parts, its [data] and [signature]. The
 * [signature] is specific to the [data] and RSA key such that if it
 * or the [signature] are modified, the [ApiKey] is invalidated.
 *
 * The [signature] is always verified upon instantiation of [ApiKey]; a
 * valid [signature] for [data] **must** be provided to instantiate [ApiKey].
 *
 * @see [create]
 * @see [createApiKey]
 * @see [Data]
 * @see [Property]
 * */
public open class ApiKey<P: ApiKey.Property> private constructor(

    /**
     * The [Data] associated with this [ApiKey]
     * */
    @JvmField
    public val data: Data<P>,

    // Always want to restore the same bytes that were
    // used when creating the signature, as data.toString()
    // output may change which, if used to verify the signature,
    // would produce a false positive. This also ensures that
    // further invocation of ApiKey.encode() produces the exact
    // same results as when the key was originally created.
    private val dataBytes: ByteArray,
    private val rsaSignature: RSASignature,
) {

    /**
     * Alternative constructor for inheritors of [ApiKey] which
     * reconstructs [Data] and verifies the RSA signature.
     *
     * @param [encodedApiKey] the Base64 encoding output by [ApiKey.encode]
     * @param [rsaPublicKey] the Base64 encoded PKCS#1 or X509 RSA Public Key
     *   to verify the signature with.
     * @param [decoder] The [Property.Decoder] to reconstruct [Data] with
     * @throws [IllegalArgumentException] when:
     *  - [encodedApiKey] is not valid
     *  - [rsaPublicKey] is not a valid Base64 encoded RSA Public Key
     *  - Signature verification fails
     * */
    @Throws(IllegalArgumentException::class)
    protected constructor(
        encodedApiKey: String,
        rsaPublicKey: String,
        decoder: Property.Decoder<P>,
    ): this(
        dataBytes = encodedApiKey
            .substringBefore(':', missingDelimiterValue = "." /* FAIL */)
            .decodeToByteArrayOrNull(Base64.Default)
            ?: throw IllegalArgumentException("Invalid ApiKey encoding"),
        rsaSignature = try {
            RSASignature.of(base64 = encodedApiKey.substringAfter(':'))
        } catch (e: EncodingException) {
            throw IllegalArgumentException("Invalid ApiKey encoding", e)
        },
        rsaPublicKey = rsaPublicKey,
        decoder = decoder,
    )

    @get:JvmName("keyId")
    public inline val keyId: String get() = data.keyId
    @get:JvmName("uuid")
    public inline val uuid: String get() = data.uuid
    @get:JvmName("properties")
    public inline val properties: Set<P> get() = data.properties
    @get:JvmName("extras")
    public inline val extras: Set<String> get() = data.extras
    @get:JvmName("createdAt")
    public inline val createdAt: String get() = data.createdAt
    @get:JvmName("expiresAt")
    public inline val expiresAt: String get() = data.expiresAt
    @Suppress("NOTHING_TO_INLINE")
    public inline fun expiresIn(): Duration = data.expiresIn()
    @Suppress("NOTHING_TO_INLINE")
    public inline fun expiresIn(clock: Clock): Duration = data.expiresIn(clock)
    @Suppress("NOTHING_TO_INLINE")
    public inline fun isExpired(): Boolean = data.isExpired()
    @Suppress("NOTHING_TO_INLINE")
    public inline fun isExpired(clock: Clock): Boolean = data.isExpired(clock)
    @Suppress("NOTHING_TO_INLINE")
    public inline fun wasCreatedAfter(instant: Instant): Boolean = data.wasCreatedAfter(instant)
    @Suppress("NOTHING_TO_INLINE")
    public inline fun wasCreatedAfter(epochMillis: Long): Boolean = data.wasCreatedAfter(epochMillis)

    /**
     * The Base64 encoded RSA signature for the SHA3-256 hash value
     * of [data] using signature algorithm NONEwithRSA.
     * */
    public fun signature(): String = rsaSignature.encoded()

    /**
     * Encodes the [ApiKey] by concatenating the [data] with the
     * [signature].
     *
     * <base64 encoded data bytes>:<base64 encoded signature bytes>
     * */
    public fun encode(): String {
        val sb = StringBuilder()

        val out = LineBreakOutFeed(interval = 64) { char -> sb.append(char) }

        Base64.Default.newEncoderFeed(out = out).use { feed ->
            dataBytes.forEach { b -> feed.consume(b) }
            feed.flush()
            out.output(':')
            rsaSignature.bytes().forEach { b -> feed.consume(b) }
        }

        return sb.toString()
    }

    public companion object {

        /**
         * Creates a new [ApiKey] for the provided arguments.
         *
         * @param [rsaPrivateKey] The unencrypted PKCS#1 or PKCS#8
         *   RSA Private Key to sign with.
         * @param [fillPrivateKey] if true, will zero out the [rsaPrivateKey]
         *   after creating the [ApiKey]
         * @throws [IllegalArgumentException] when:
         *  - [rsaPrivateKey] is not a valid RSA Private Key
         *  - [keyId] is blank
         *  - [keyId] contains white space
         *  - [keyId] contains multiple lines
         *  - [daysUntilExpiry] are less than or equal to 0
         *  - Any [extras] provided are blank/empty
         *  - Any [extras] provided contain multiple lines
         * */
        @JvmStatic
        @JvmOverloads
        @Throws(IllegalArgumentException::class)
        public fun <P: Property> create(
            rsaPrivateKey: ByteArray,
            fillPrivateKey: Boolean,
            keyId: String,
            properties: Set<P>,
            daysUntilExpiry: Int,
            extras: Set<String> = emptySet(),
        ): ApiKey<P> {
            val result = try {
                val data = Data.create(keyId, properties, daysUntilExpiry, extras)

                RSAScope.Private.useDecrypted(rsaPrivateKey) { createApiKey(data) }
            } catch (e: GeneralSecurityException) {
                throw IllegalArgumentException("Invalid RSA Private Key", e)
            } finally {
                if (fillPrivateKey) rsaPrivateKey.fill(0)
            }

            return result
        }

        /**
         * Creates a new [ApiKey] for the provided arguments.
         *
         * This extension function requires a dependency.
         * See: https://github.com/toxicity-io/crypto
         *
         * @throws [IllegalArgumentException] when:
         *  - [keyId] is blank
         *  - [keyId] contains white space
         *  - [keyId] contains multiple lines
         *  - [daysUntilExpiry] are less than or equal to 0
         *  - Any [extras] provided are blank/empty
         *  - Any [extras] provided contain multiple lines
         * */
        @JvmStatic
        @JvmOverloads
        @Throws(GeneralSecurityException::class, IllegalArgumentException::class)
        public fun <P: Property> RSAScope.Private.createApiKey(
            keyId: String,
            properties: Set<P>,
            daysUntilExpiry: Int,
            extras: Set<String> = emptySet(),
        ): ApiKey<P> = createApiKey(Data.create(keyId, properties, daysUntilExpiry, extras))

        @JvmStatic
        private fun <P: Property> RSAScope.Private.createApiKey(
            data: Data<P>,
        ): ApiKey<P> = data.toByteArray().let { bytes -> ApiKey(data, bytes, bytes.sign()) }
    }

    public class Data<P: Property> private constructor(

        /**
         * The id associated with this [ApiKey].
         * */
        @JvmField
        public val keyId: String,

        /**
         * The unique id for this [ApiKey].
         *
         * e.g. VV5A9-HCSPY-HGDQR-XJG0S-XY6P9-3TN09-XYNMZ-TCFWH-TSBJJ
         * */
        @JvmField
        public val uuid: String,

        /**
         * A set of properties specific to the [ApiKey]'s implementation.
         * */
        @JvmField
        public val properties: Set<P>,

        /**
         * Any extra information that may be needed which does not fit
         * as a [Property].
         * */
        @JvmField
        public val extras: Set<String>,

        /**
         * The time at which the [Data] was first created.
         *
         * Format: ISO 8601 international standard, with timezone UTC
         * */
        @JvmField
        public val createdAt: String,

        /**
         * The time at which the [Data], and thus the [ApiKey] will expire.
         *
         * Format: ISO 8601 international standard, with timezone UTC
         * */
        @JvmField
        public val expiresAt: String,
    ) {

        public fun expiresIn(): Duration = expiresIn(Clock.System)
        public fun expiresIn(clock: Clock): Duration = expiresAt.toInstant().minus(clock.now())

        public fun isExpired(): Boolean = isExpired(Clock.System)
        public fun isExpired(clock: Clock): Boolean = expiresIn(clock).isNegative()

        public fun wasCreatedAfter(instant: Instant): Boolean = createdAt.toInstant() > instant

        @Throws(Throwable::class)
        public fun wasCreatedAfter(epochMillis: Long): Boolean {
            val instant = Instant.fromEpochMilliseconds(epochMillis)
            return wasCreatedAfter(instant)
        }

        public override fun equals(other: Any?): Boolean = other is Data<*> && other.uuid == uuid
        public override fun hashCode(): Int = 17 * 31 + uuid.hashCode()
        public override fun toString(): String = StringBuilder().apply {
            joinTo(builder = this, prettyPrint = true)
        }.toString()

        @JvmSynthetic
        internal fun toByteArray(): ByteArray = StringBuilder().apply {
            joinTo(builder = this, prettyPrint = false)
        }.toString().encodeToByteArray()

        public companion object {

            // NEVER change this. Used for encoding/decoding.
            internal const val KEY_ID = "keyId: "

            /**
             * Decodes the [Data] portion of the encoded [ApiKey] that
             * was output from [ApiKey.encode].
             * */
            @JvmStatic
            @Throws(RuntimeException::class)
            public fun decodeToString(encodedApiKey: String): String = encodedApiKey
                .substringBefore(':')
                .decodeToByteArray(Base64.Default)
                .decodeToString()

            @JvmStatic
            @JvmSynthetic
            @Throws(IllegalArgumentException::class)
            internal fun <P: Property> create(
                keyId: String,
                properties: Set<P>,
                daysUntilExpiry: Int,
                extras: Set<String>,
                clock: Clock = Clock.System
            ): Data<P> {
                require(keyId.isNotBlank()) { "keyId cannot be blank" }
                require(!keyId.contains(' ')) { "keyId cannot contain whitespace" }
                require(keyId.lines().size == 1) { "keyId must be a single line" }

                require(daysUntilExpiry > 0) { "daysUntilExpiry must be greater than 0" }

                for (extra in extras) {
                    require(extra.isNotBlank()) { "extras cannot be blank" }
                    require(extra.lines().size == 1) { "extras must be a single line" }
                }

                val now = clock.now().toEpochMilliseconds().let { Instant.fromEpochMilliseconds(it) }
                val expiresAt = now.plus(daysUntilExpiry.days).let { instant ->
                    if (instant > Instant.DISTANT_FUTURE) Instant.DISTANT_FUTURE else instant
                }

                val uuid = uuid(keyId, properties, extras, now.toString(), expiresAt.toString())

                return Data(
                    keyId,
                    uuid,
                    properties,
                    extras,
                    now.toString(),
                    expiresAt.toString(),
                )
            }

            @JvmStatic
            @JvmSynthetic
            @Throws(IllegalArgumentException::class)
            internal fun <P: Property> ByteArray.toApiKeyData(decoder: Property.Decoder<P>): Data<P> {
                val lines = decodeToString().lines()

                require(lines.size >= 6) { "Failed to decode data string" }

                var i = 0
                val keyId = lines.getOrThrow(i++).let {
                    require(it.startsWith(KEY_ID)) { "Failed to decode data string" }
                    it.substringAfter(KEY_ID)
                }

                val uuid = lines.getOrThrow(i++).substringAfter(": ")

                val properties = buildSet {
                    var lastLine = lines.getOrThrow(i++)

                    // Check for empty properties
                    if (!lastLine.endsWith(']')) {
                        lastLine = lines.getOrThrow(i++) // skip "properties: ["

                        while (!lastLine.endsWith(']')) {
                            val property = decoder.decode(lastLine.substringAfter("    "))
                            add(property)
                            lastLine = lines.getOrThrow(i++)
                        }
                    }
                }

                val extras = buildSet {
                    var lastLine = lines.getOrThrow(i++)

                    // Check for empty extras
                    if (!lastLine.endsWith(']')) {
                        lastLine = lines.getOrThrow(i++) // skip "extras: ["

                        while (!lastLine.endsWith(']')) {
                            val extra = lastLine.substringAfter("    ")
                            add(extra)
                            lastLine = lines.getOrThrow(i++)
                        }
                    }
                }

                val createdAt = lines.getOrThrow(i++).substringAfter(": ")
                val expiresAt = lines.getOrThrow(i).substringAfter(": ")

                val uuidCheck = uuid(keyId, properties, extras, createdAt, expiresAt)

                if (uuid != uuidCheck) {
                    throw IllegalArgumentException(
                        "Invalid uuid: Expected[$uuidCheck] vs Actual[$uuid]. Was data modified?"
                    )
                }

                return Data(
                    keyId,
                    uuid,
                    properties,
                    extras,
                    createdAt,
                    expiresAt,
                )
            }

            @JvmStatic
            private fun <P: Property> uuid(
                keyId: String,
                properties: Set<P>,
                extras: Set<String>,
                createdAt: String,
                expiresAt: String,
            ): String = SHA3_224().apply {
                update(keyId.encodeToByteArray())
                properties.forEach { p -> update(p.encoding.encodeToByteArray()) }
                extras.forEach { e -> update(e.encodeToByteArray()) }
                update(createdAt.encodeToByteArray())
                update(expiresAt.encodeToByteArray())
            }.digest().encodeToString(Base32Crockford { hyphenInterval = 5 })
        }
    }

    /**
     * Abstraction for modeling type safe properties of the [ApiKey] implementation.
     *
     * Implementations should also model an "Unknown" value in the event new
     * properties are added to expand the api key's functionality.
     *
     * e.g.
     *
     *     public sealed class MyProperty(encoding: String): ApiKey.Property(encoding) {
     *         public object AUTHENTICATION: MyProperty("AUTHENTICATION")
     *         public class Something(public val count: Int): MyProperty("SOMETHING:$count")
     *         public class Unknown internal constructor(value: String): MyProperty(value)
     *
     *         public companion object: Decoder<MyProperty>() {
     *
     *             @Throws(IllegalArgumentException::class)
     *             override fun decode(value: String): MyProperty {
     *                 return when {
     *                     value == "AUTHENTICATION" -> AUTHENTICATION
     *                     value.startsWith("SOMETHING:") -> Something(value.substringAfter(':').toInt())
     *                     else -> Unknown(value)
     *                 }
     *             }
     *         }
     *     }
     *
     * @throws [IllegalArgumentException] when:
     *  - [encoding] is blank/empty
     *  - [encoding] contains multiple lines
     * */
    public abstract class Property
    @Throws(IllegalArgumentException::class)
    protected constructor(
        @JvmField
        public val encoding: String,
    ) {

        init {
            require(encoding.isNotBlank()) { "${this::class.simpleName}.encoding cannot be blank" }
            require(encoding.lines().size == 1) { "${this::class.simpleName}.encoding cannot contain multiple lines" }
        }

        final override fun equals(other: Any?): Boolean = other is Property && other.encoding == encoding
        final override fun hashCode(): Int = 17 * 31 + encoding.hashCode()

        /**
         * A decoder for the [ApiKey] implementation's properties. Used
         * to rebuild the [Data] from the Base64 encoded [ApiKey] string.
         * */
        public abstract class Decoder<T: Property> {
            @Throws(IllegalArgumentException::class)
            public abstract fun decode(value: String): T
        }
    }

    private constructor(
        dataBytes: ByteArray,
        rsaSignature: RSASignature,
        rsaPublicKey: String,
        decoder: Property.Decoder<P>
    ): this(
        data = dataBytes.toApiKeyData(decoder),
        dataBytes = dataBytes,
        rsaSignature = rsaSignature,
    ) {
        try {
            RSAScope.Public.useEncoded(rsaPublicKey) {
                if (!dataBytes.verify(rsaSignature)) {
                    throw IllegalArgumentException("Invalid RSA Signature")
                }
            }
        } catch (e: GeneralSecurityException) {
            throw IllegalArgumentException("Invalid RSA Public Key", e)
        }
    }

    public final override fun equals(other: Any?): Boolean {
        return  other is ApiKey<*>
                && other.data == data
                && other.rsaSignature == rsaSignature
    }

    public final override fun hashCode(): Int {
        var result = 17
        result = result * 31 + data.hashCode()
        result = result * 31 + rsaSignature.hashCode()
        return result
    }

    public final override fun toString(): String {
        return StringBuilder().apply {
            data.joinTo(builder = this, prettyPrint = true)
            appendLine()
            append("Signature: [")
            appendLine()
            signature().lines().joinTo(this, separator = "\n    ", prefix = "    ")
            appendLine()
            append(']')
        }.toString()
    }

}

@Suppress("NOTHING_TO_INLINE")
@Throws(IllegalArgumentException::class)
private inline fun List<String>.getOrThrow(index: Int): String {
    return try {
        this[index]
    } catch (e: IndexOutOfBoundsException) {
        throw IllegalArgumentException("Failed to decode data", e)
    }
}

private fun ApiKey.Data<*>.joinTo(builder: StringBuilder, prettyPrint: Boolean) {
    val indent = if (prettyPrint) "    " else ""
    val prefix = if (prettyPrint) "        " else "    "
    val separator = "\n$prefix"

    builder.apply {
        if (prettyPrint) {
            append("Data: [")
            appendLine()
        }

        append(indent)
        append(ApiKey.Data.KEY_ID)
        append(keyId)

        appendLine()
        append(indent)
        append("uuid: ")
        append(uuid)

        appendLine()
        append(indent)
        append("properties: [")
        if (properties.isNotEmpty()) {
            appendLine()
            properties.joinTo(this, separator = separator, prefix = prefix) { it.encoding }
            appendLine()
            append(indent)
            append(']')
        } else {
            append(']')
        }

        appendLine()
        append(indent)
        append("extras: [")
        if (extras.isNotEmpty()) {
            appendLine()
            extras.joinTo(this, separator = separator, prefix = prefix) { it }
            appendLine()
            append(indent)
            append(']')
        } else {
            append(']')
        }

        appendLine()
        append(indent)
        append("createdAt: ")
        append(createdAt)

        appendLine()
        append(indent)
        append("expiresAt: ")
        append(expiresAt)

        if (prettyPrint) {
            appendLine()
            append(']')
        }
    }
}
