/*
 * Copyright (c) 2023 Toxicity
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
@file:Suppress("KotlinRedundantDiagnosticSuppress", "NOTHING_TO_INLINE")

package io.toxicity.api.key.internal

import io.toxicity.api.key.ApiKey
import kotlin.time.Clock
import kotlin.time.Duration
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

internal inline fun CharSequence.isSingleLine(): Boolean {
    return indexOfFirst { char -> char == '\r' || char == '\n' } == -1
}

internal inline fun CharSequence.hasWhitespace(): Boolean {
    return indexOfFirst { char -> char.isWhitespace() } != -1
}

internal inline fun String.isValidUtf8(): Boolean = try {
    encodeToByteArray(throwOnInvalidSequence = true)
    true
} catch (_: CharacterCodingException) {
    false
}

@ExperimentalTime
internal inline fun ApiKey.Data<*>.expiresIn(clock: Clock): Duration {
    return Instant.parse(expiresAt).minus(clock.now())
}

@ExperimentalTime
internal inline fun ApiKey.Data<*>.isExpired(clock: Clock): Boolean {
    return expiresIn(clock).isNegative()
}

@ExperimentalTime
internal inline fun ApiKey.Data<*>.wasCreatedAfter(instant: Instant): Boolean {
    return Instant.parse(createdAt) > instant
}
