/*
 * Copyright (c) 2023 Toxicity
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **/
package io.toxicity.api.key

import io.matthewnelson.encoding.base32.Base32Crockford
import io.matthewnelson.encoding.base64.Base64
import io.matthewnelson.encoding.core.Decoder.Companion.decodeToByteArray
import io.matthewnelson.encoding.core.Encoder.Companion.encodeToString
import io.matthewnelson.encoding.core.EncodingException
import io.matthewnelson.encoding.core.use
import io.matthewnelson.encoding.core.util.LineBreakOutFeed
import io.matthewnelson.immutable.collections.toImmutableSet
import io.toxicity.api.key.internal.*
import io.toxicity.api.key.internal.hasWhitespace
import io.toxicity.api.key.internal.isExpired
import io.toxicity.api.key.internal.isSingleLine
import io.toxicity.api.key.internal.isValidUtf8
import io.toxicity.api.key.ApiKey.Data.Companion.toApiKeyData
import io.toxicity.crypto.core.DestroyedStateException
import io.toxicity.crypto.core.scope.CryptoKeyScope.Factory.cryptoKeyScope
import io.toxicity.crypto.rsa.RSAKey
import io.toxicity.crypto.rsa.RSAKey.Private.Companion.rsaPrivateKeyOf
import io.toxicity.crypto.rsa.RSAKey.Public.Companion.rsaPublicKeyOf
import io.toxicity.crypto.rsa.RSASignature
import org.kotlincrypto.error.GeneralSecurityException
import org.kotlincrypto.error.InvalidKeyException
import org.kotlincrypto.error.InvalidParameterException
import org.kotlincrypto.error.requireParam
import org.kotlincrypto.hash.sha3.SHA3_224
import org.kotlincrypto.hash.sha3.SHA3_256
import kotlin.jvm.JvmField
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlin.jvm.JvmSynthetic
import kotlin.time.Clock
import kotlin.time.Duration
import kotlin.time.Duration.Companion.days
import kotlin.time.ExperimentalTime
import kotlin.time.Instant

/**
 * An [ApiKey] consists of 2 parts, its [data] and [signature]. The
 * [signature] is specific to the [data] and RSA key such that if it
 * or the [signature] are modified, the [ApiKey] is invalidated.
 *
 * The [signature] is always verified upon instantiation of [ApiKey]; a
 * valid [signature] for [data] **must** be provided to instantiate [ApiKey].
 *
 * @see [create]
 * @see [createApiKey]
 * @see [Data]
 * @see [Property]
 * */
public open class ApiKey<P: ApiKey.Property> {

    /**
     * The [Data] associated with this [ApiKey]
     * */
    @JvmField
    public val data: Data<P>

    // Always want to restore the same bytes that were used when the
    // RSASignature was created, as data.toString() output may change
    // such that if it were to be used to verify the signature, would
    // result in a false positive. This also ensures that further
    // invocations of ApiKey.encoded() produce the exact same results as
    // when the key was originally created.
    private val _data: ByteArray
    private val _signature: RSASignature

    /**
     * Instantiates a new [ApiKey] instance, verifying the associated [RSASignature] within
     * provided [encodedApiKey], using the provided [RSAKey.Public]. [Data] is subsequently
     * reconstructed from [encodedApiKey].
     *
     * @param [encodedApiKey] the resultant text output by [ApiKey.encoded]
     * @param [publicKey] the base64 encoded [RSAKey.Public]
     * @param [decoder] the [Property.Decoder] associated with this [ApiKey]
     *
     * @throws [IllegalArgumentException] if [Property.Decoder.decode] fails
     * @throws [EncodingException] if [encodedApiKey] is not valid, or base64 decoding fails
     * @throws [GeneralSecurityException] if [publicKey] or the [RSASignature] is invalid
     * */
    protected constructor(
        encodedApiKey: String,
        publicKey: String,
        decoder: Property.Decoder<P>,
    ): this(decoder, encodedApiKey, publicKey)

    /**
     * Encodes the [ApiKey] by concatenating its encoded [Data] bytes that are base64
     * encoded, with the base64 encoding of its associated [RSASignature], separated
     * by a delimiter character of `:`.
     * */
    public fun encoded(): String {
        val sig = _signature.bytes()
        val sb = Base64.Default.config.encodeOutSize(
            unEncodedSize = (sig.size + _data.size).toLong()
        ).let { size -> StringBuilder(size.toInt() + 10) }

        val out = LineBreakOutFeed(Base64.Default.config.lineBreakInterval) { char -> sb.append(char) }

        Base64.Default.newEncoderFeed(out).use { feed ->
            _data.forEach { b -> feed.consume(b) }
            feed.flush()
            out.output(':')
            sig.forEach { b -> feed.consume(b) }
        }

        return sb.toString()
    }

    /**
     * The Base64 encoded [RSASignature] for the [SHA3_256] hash value of
     * [data] using signature algorithm `NoneWithRSA`.
     * */
    public fun signature(): String = _signature.base64()

    public companion object {

        // NEVER change this value. Used for encoding/decoding.
        private const val INDENT = "    "

        /**
         * Creates a new [ApiKey] for the provided arguments.
         *
         * @param [privateKey] the raw, unencrypted PKCS#1 or PKCS#8 encoded [RSAKey.Private] bytes
         * @param [clear] if `true`, [privateKey] bytes will be zeroed out
         *
         * @throws [GeneralSecurityException] if [privateKey] is invalid
         * @throws [InvalidKeyException] if [privateKey] is invalid
         * @throws [InvalidParameterException] when:
         *   - [keyId] is blank
         *   - [keyId] contains whitespace
         *   - [keyId] contains multiple lines
         *   - [keyId] contains invalid UTF-8 characters or sequences
         *   - [daysUntilExpiry] is less than or equal to 0
         *   - [extras] contains a blank value
         *   - [extras] contains a value with multiple lines
         *   - [extras] contains a value with invalid UTF-8 characters or sequences
         * */
        @JvmStatic
        @JvmOverloads
        public fun <P: Property> create(
            privateKey: ByteArray,
            clear: Boolean,
            keyId: String,
            properties: Set<P>,
            daysUntilExpiry: Int,
            extras: Set<String> = emptySet(),
        ): ApiKey<P> = cryptoKeyScope(initialCapacity = 1) {
            rsaPrivateKeyOf(unencrypted = privateKey, clear = clear)
                .createApiKey(keyId, properties, daysUntilExpiry, extras)
        }

        /**
         * Creates a new [ApiKey] for provided arguments.
         *
         * **NOTE:** This extension function requires a dependency.
         *
         * See: https://github.com/toxicity-io/crypto
         *
         * @throws [DestroyedStateException] if the [RSAKey.Private] is destroyed
         * @throws [InvalidParameterException] when:
         *   - [keyId] is blank
         *   - [keyId] contains whitespace
         *   - [keyId] contains multiple lines
         *   - [keyId] contains invalid UTF-8 characters or sequences
         *   - [daysUntilExpiry] is less than or equal to 0
         *   - [extras] contains a blank value
         *   - [extras] contains a value with multiple lines
         *   - [extras] contains a value with invalid UTF-8 characters or sequences
         * */
        @JvmStatic
        @JvmOverloads
        public fun <P: Property> RSAKey.Private.createApiKey(
            keyId: String,
            properties: Set<P>,
            daysUntilExpiry: Int,
            extras: Set<String> = emptySet(),
        ): ApiKey<P> {
            @OptIn(ExperimentalTime::class)
            val data = Data.create(keyId, properties, daysUntilExpiry, extras)
            val encoded = data.toString().encodeToByteArray(throwOnInvalidSequence = true)
            val hash = SHA3_256().digest(encoded)
            val signature = sign(hash)
            return ApiKey(data, encoded, signature)
        }
    }

    public class Data<P: Property> private constructor(

        /**
         * The id associated with this [ApiKey]
         * */
        @JvmField
        public val keyId: String,

        /**
         * The unique id for this [ApiKey]
         *
         * e.g.
         *
         *     VV5A9-HCSPY-HGDQR-XJG0S-XY6P9-3TN09-XYNMZ-TCFWH-TSBJJ
         * */
        @JvmField
        public val uuid: String,

        /**
         * A set of properties specific to the [ApiKey] implementation
         * */
        @JvmField
        public val properties: Set<P>,

        /**
         * Any extra information that may be needed, but is not suitable as a [Property]
         * */
        @JvmField
        public val extras: Set<String>,

        /**
         * The ISO 8601 formatted time at which the [Data] was first created
         * */
        @JvmField
        public val createdAt: String,

        /**
         * The ISO 8601 formatted time at which [Data] (and thus [ApiKey]) will expire
         * */
        @JvmField
        public val expiresAt: String,
    ) {

        public fun expiresIn(): Duration {
            @OptIn(ExperimentalTime::class)
            return expiresIn(Clock.System)
        }

        public fun isExpired(): Boolean {
            @OptIn(ExperimentalTime::class)
            return isExpired(Clock.System)
        }

        public fun wasCreatedAfter(epochMillis: Long): Boolean {
            @OptIn(ExperimentalTime::class)
            return wasCreatedAfter(Instant.fromEpochMilliseconds(epochMillis))
        }

        public companion object {

            // NEVER change these values. Used for encoding/decoding.
            private const val FIELD_KEY_ID = "keyId: "
            private const val FIELD_UUID = "uuid: "
            private const val FIELD_PROPERTIES = "properties: ["
            private const val FIELD_EXTRAS = "extras: ["
            private const val FIELD_CREATED_AT = "createdAt: "
            private const val FIELD_EXPIRES_AT = "expiresAt: "

            /**
             * Decodes the [Data] portion of the encoded [ApiKey] output by [ApiKey.encoded]
             *
             * @throws [EncodingException] if [encodedApiKey] contains invalid base64 characters
             * */
            @JvmStatic
            public fun decodeToString(encodedApiKey: String): String = encodedApiKey
                .substringBefore(':')
                .decodeToByteArray(Base64.Default)
                .decodeToString()

            @JvmSynthetic
            @OptIn(ExperimentalTime::class)
            @Throws(InvalidParameterException::class)
            internal fun <P: Property> create(
                keyId: String,
                properties: Set<P>,
                daysUntilExpiry: Int,
                extras: Set<String>,
                clock: Clock = Clock.System,
            ): Data<P> {
                requireParam(keyId.isNotBlank()) { "keyId cannot be blank" }
                requireParam(!keyId.hasWhitespace()) { "keyId cannot contain whitespace" }
                requireParam(keyId.isValidUtf8()) { "keyId cannot contain invalid UTF-8 characters or sequences" }

                requireParam(daysUntilExpiry > 0) { "daysUntilExpiry must be greater than 0" }

                val iExtras = extras.toImmutableSet()
                for (extra in iExtras) {
                    requireParam(extra.isNotBlank()) { "extras cannot be blank" }
                    requireParam(extra.isSingleLine()) { "extras cannot contain multiple lines" }
                    requireParam(extra.isValidUtf8()) { "extras cannot contain invalid UTF-8 characters or sequences" }
                }

                val createdAt = clock.now().toEpochMilliseconds().let { Instant.fromEpochMilliseconds(it) }
                val expiresAt = createdAt.plus(daysUntilExpiry.days).let { instant ->
                    if (instant > Instant.DISTANT_FUTURE) Instant.DISTANT_FUTURE else instant
                }

                val sCreatedAt = createdAt.toString()
                val sExpiresAt = expiresAt.toString()
                val iProperties = properties.toImmutableSet()
                val uuid = uuid(keyId, iProperties, iExtras, sCreatedAt, sExpiresAt)

                return Data(keyId, uuid, iProperties, iExtras, sCreatedAt, sExpiresAt)
            }

            @JvmSynthetic
            @Throws(EncodingException::class, GeneralSecurityException::class, IllegalArgumentException::class)
            internal fun <P: Property> ByteArray.toApiKeyData(decoder: Property.Decoder<P>): Data<P> {
                val lines = try {
                    decodeToString(throwOnInvalidSequence = true)
                } catch (e: CharacterCodingException) {
                    throw EncodingException("Encoded ApiKey.Data contains invalid UTF-8 character sequences", e)
                }.lines()

                if (lines.size < 6) {
                    throw EncodingException("Encoded ApiKey.Data must be greater than 5 lines of text")
                }

                var i = 0
                val keyId = lines[i++].let { line ->
                    if (!line.startsWith(FIELD_KEY_ID)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_KEY_ID' field")
                    }
                    line.substring(FIELD_KEY_ID.length, line.length)
                }

                val uuid = lines[i++].let { line ->
                    if (!line.startsWith(FIELD_UUID)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_UUID' field")
                    }
                    line.substring(FIELD_UUID.length, line.length)
                }

                val properties = LinkedHashSet<P>(5, 1.0F).apply {
                    var line = lines[i++]

                    if (!line.startsWith(FIELD_PROPERTIES)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_PROPERTIES' field")
                    }

                    // Check for empty properties
                    if (line.endsWith(']')) return@apply

                    line = lines[i++]
                    while (i < lines.size && line != "]") {
                        if (!line.startsWith(INDENT)) {
                            throw EncodingException("Encoded ApiKey.Data.property line must start with an indent")
                        }
                        val encoding = line.substring(INDENT.length, line.length)
                        val property = try {
                            decoder.decode(encoding)
                        } catch (t: Throwable) {
                            if (t is IllegalArgumentException) throw t
                            throw IllegalArgumentException("ApiKey.Property.Decoder.decode failed", t)
                        }
                        add(property)
                        line = lines[i++]
                    }
                }.toImmutableSet()

                // There should be a minimum of 3 lines left for fields extras, createdAt, and expiresAt
                if (lines.size - i < 3) {
                    throw EncodingException("Encoded ApiKey.Data truncated. Not enough lines remaining.")
                }

                val extras = LinkedHashSet<String>(5, 1.0F).apply {
                    var line = lines[i++]

                    if (!line.startsWith(FIELD_EXTRAS)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_EXTRAS' field")
                    }

                    // Check for empty extras
                    if (line.endsWith(']')) return@apply

                    line = lines[i++]
                    while (i < lines.size && line != "]") {
                        if (!line.startsWith(INDENT)) {
                            throw EncodingException("Encoded ApiKey.Data.extras line must start with an indent")
                        }
                        add(line.substring(INDENT.length, line.length))
                        line = lines[i++]
                    }
                }.toImmutableSet()

                // There should be a minimum of 2 lines left for fields createdAt and expiresAt
                if (lines.size - i < 2) {
                    throw EncodingException("Encoded ApiKey.Data truncated. Not enough lines remaining.")
                }

                val createdAt = lines[i++].let { line ->
                    if (!line.startsWith(FIELD_CREATED_AT)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_CREATED_AT' field")
                    }
                    line.substring(FIELD_CREATED_AT.length, line.length)
                }

                val expiresAt = lines[i++].let { line ->
                    if (!line.startsWith(FIELD_EXPIRES_AT)) {
                        throw EncodingException("Encoded ApiKey.Data must contain a '$FIELD_EXPIRES_AT' field")
                    }
                    line.substring(FIELD_EXPIRES_AT.length, line.length)
                }

                val uuidCalculated = uuid(keyId, properties, extras, createdAt, expiresAt)

                if (uuid != uuidCalculated) {
                    throw GeneralSecurityException("Invalid uuid >> Decoded[$uuid] vs Calculated[$uuidCalculated]")
                }

                return Data(keyId, uuid, properties, extras, createdAt, expiresAt)
            }

            @Throws(CharacterCodingException::class)
            private fun uuid(
                keyId: String,
                properties: Set<Property>,
                extras: Set<String>,
                createdAt: String,
                expiresAt: String,
            ): String = SHA3_224().apply {
                update(keyId.encodeToByteArray(throwOnInvalidSequence = true))
                properties.forEach { p -> update(p.encoding.encodeToByteArray(throwOnInvalidSequence = true)) }
                extras.forEach { e -> update(e.encodeToByteArray(throwOnInvalidSequence = true)) }
                update(createdAt.encodeToByteArray(throwOnInvalidSequence = true))
                update(expiresAt.encodeToByteArray(throwOnInvalidSequence = true))
            }.digest().encodeToString(CROCKFORD)

            private val CROCKFORD = Base32Crockford { hyphenInterval = 5 }
        }

        /** @suppress */
        public override fun equals(other: Any?): Boolean = other is Data<*> && other.uuid == uuid
        /** @suppress */
        public override fun hashCode(): Int = 17 * 31 + uuid.hashCode()
        /** @suppress */
        public override fun toString(): String = buildString {
            append(FIELD_KEY_ID).appendLine(keyId)
            append(FIELD_UUID).appendLine(uuid)
            append(FIELD_PROPERTIES)
            if (properties.isNotEmpty()) {
                appendLine()
                properties.forEach { property ->
                    append(INDENT).appendLine(property.encoding)
                }
            }
            appendLine(']')
            append(FIELD_EXTRAS)
            if (extras.isNotEmpty()) {
                appendLine()
                extras.forEach { extra ->
                    append(INDENT).appendLine(extra)
                }
            }
            appendLine(']')
            append(FIELD_CREATED_AT).appendLine(createdAt)
            append(FIELD_EXPIRES_AT).append(expiresAt)
        }
    }

    /**
     * Abstraction for modeling type-safe properties of the [ApiKey] implementation.
     *
     * Implementations should also model an "Unknown" value in the event new properties
     * are added, such that older versions of the implementation remain functional.
     *
     * e.g.
     *
     *     public abstract class MyProperty private constructor(
     *         encoding: String,
     *     ): ApiKey.Property(encoding) {
     *
     *         public data object AUTHENTICATION: MyProperty("AUTHENTICATION")
     *         public class Something(public val count: Int): MyProperty("SOMETHING:$count")
     *         public class Unknown internal constructor(value: String): MyProperty(value)
     *
     *         public companion object: Decoder<MyProperty>() {
     *
     *             @Throws(IllegalArgumentException::class)
     *             public override fun decode(value: String): MyProperty = when {
     *                 value == AUTHENTICATION.encoding -> AUTHENTICATION
     *                 value.startsWith("SOMETHING") -> Something(value.substringAfter(':').toInt())
     *                 else -> Unknown(value)
     *             }
     *         }
     *     }
     *
     * @throws [IllegalArgumentException] when:
     *   - Implementing class is an anonymous object
     *   - [encoding] is blank
     *   - [encoding] is multiple lines
     *   - [encoding] contains invalid UTF-8 characters or sequences
     * */
    public abstract class Property
    @Throws(IllegalArgumentException::class)
    protected constructor(
        @JvmField
        public val encoding: String,
    ) {

        /**
         * An abstraction for transforming a property's [encoding] value to its [Property] type.
         * */
        public abstract class Decoder<P: Property> protected constructor() {
            @Throws(IllegalArgumentException::class)
            public abstract fun decode(encoding: String): P
        }

        init {
            requireNotNull(this::class.qualifiedName) { "Property cannot be an anonymous object" }
            require(encoding.isNotBlank()) { "encoding cannot be blank" }
            require(encoding.isSingleLine()) { "encoding cannot contain multiple lines" }
            require(encoding.isValidUtf8()) { "encoding cannot contain invalid UTF-8 characters or sequences" }
        }

        /** @suppress */
        public final override fun equals(other: Any?): Boolean {
            if (other !is Property) return false
            if (other::class != this::class) return false
            return other.encoding == this.encoding
        }
        /** @suppress */
        public final override fun hashCode(): Int {
            var result = 17
            result = result * 31 + this::class.hashCode()
            result = result * 31 + encoding.hashCode()
            return result
        }
        /** @suppress */
        public override fun toString(): String {
            val name: String = this::class.qualifiedName?.let { qn ->
                // some.package.name.ClassName.Subclass
                val i = qn.indexOfFirst { it.isUpperCase() }
                if (i == -1) return@let null
                qn.substring(i, qn.length)
            } ?: "ApiKey.Property"

            return "$name[encoding=$encoding]"
        }
    }

    /** @suppress */
    public override fun equals(other: Any?): Boolean {
        if (other !is ApiKey<*>) return false
        return other.data == data && other._signature == _signature
    }
    /** @suppress */
    public override fun hashCode(): Int {
        var result = 17
        result = result * 31 + data.hashCode()
        result = result * 31 + _signature.hashCode()
        return result
    }
    /** @suppress */
    public override fun toString(): String = buildString {
        appendLine("Data: [")
        data.toString().lines().forEach { line ->
            append(INDENT).appendLine(line)
        }
        appendLine(']')
        appendLine("Signature: [")
        signature().lines().forEach { line ->
            append(INDENT).appendLine(line)
        }
        append(']')
    }

    @Throws(EncodingException::class, IllegalArgumentException::class, GeneralSecurityException::class)
    private constructor(decoder: Property.Decoder<P>, encodedApiKey: String, publicKey: String) {
        val i = encodedApiKey.indexOf(':')
        if (i == -1) throw EncodingException("encodedApiKey does not contain delimiter char[:]")

        this._data = object : CharSequence {
            override val length: Int = i
            override fun get(index: Int): Char = encodedApiKey[index]
            override fun subSequence(startIndex: Int, endIndex: Int): CharSequence = error("")
        }.decodeToByteArray(Base64.Default)
        this._signature = object : CharSequence {
            override val length: Int = encodedApiKey.length - i - 1
            override fun get(index: Int): Char = encodedApiKey[i + 1 + index]
            override fun subSequence(startIndex: Int, endIndex: Int): CharSequence = error("")
        }.let { base64 -> RSASignature.of(base64) }

        cryptoKeyScope(initialCapacity = 1) {
            val key = rsaPublicKeyOf(publicKey)
            val hash = SHA3_256().digest(_data)
            if (key.verify(_signature, hash)) return@cryptoKeyScope
            // TODO: throw SignatureException
            throw GeneralSecurityException("Invalid RSA Signature")
        }

        this.data = this._data.toApiKeyData(decoder)
    }

    @Suppress("LocalVariableName")
    private constructor(data: Data<P>, _data: ByteArray, _signature: RSASignature) {
        this.data = data
        this._data = _data
        this._signature = _signature
    }
}
