/*-
 * #%L
 * IO Tesler - Liquibase
 * %%
 * Copyright (C) 2018 - 2019 Tesler Contributors
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package liquibase.ext.tesler.core;

import liquibase.change.ColumnConfig;
import liquibase.change.DatabaseChange;
import liquibase.change.core.InsertDataChange;
import liquibase.database.Database;
import liquibase.database.core.InformixDatabase;
import liquibase.ext.tesler.stmt.InsertPreparedStatement;
import liquibase.snapshot.SnapshotGeneratorFactory;
import liquibase.statement.SqlStatement;
import liquibase.statement.core.InsertStatement;
import liquibase.structure.core.Column;

@DatabaseChange(name = "insert", description = "Inserts data into an existing table", priority = Integer.MAX_VALUE)
public class InsertDataChangeCustom extends InsertDataChange {

	public InsertDataChangeCustom() {
		super();
	}

	@Override
	public SqlStatement[] generateStatements(Database database) {

		boolean needsPreparedStatement = false;
		for (ColumnConfig column : getColumns()) {
			if (column.getValueBlobFile() != null) {
				needsPreparedStatement = true;
			}
			if (column.getValueClobFile() != null) {
				needsPreparedStatement = true;
			}

			if (!needsPreparedStatement && (database instanceof InformixDatabase)) {
				if (column.getValue() != null) {
					try {
						Column snapshot = SnapshotGeneratorFactory
								.getInstance().createSnapshot(new Column(column), database);
						if (snapshot != null) {
							needsPreparedStatement = true;
						}
					} catch (Exception ignore) { //assume it's not a clob
					}
				}
			}
		}

		if (needsPreparedStatement) {
			return new SqlStatement[]{
					new InsertPreparedStatement(
							database,
							getCatalogName(),
							getSchemaName(),
							getTableName(),
							getColumns(),
							getChangeSet(),
							getResourceAccessor()
					)
			};
		}

		InsertStatement statement = new InsertStatement(getCatalogName(), getSchemaName(), getTableName());

		for (ColumnConfig column : getColumns()) {

			if (database.supportsAutoIncrement() && (column.isAutoIncrement() != null) && column.isAutoIncrement()) {
				// skip auto increment columns as they will be generated by the database
				continue;
			}

			statement.addColumnValue(column.getName(), column.getValueObject());
		}
		return new SqlStatement[]{
				statement
		};
	}


}
