/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.metadata.impl;

import io.netty.util.concurrent.DefaultThreadFactory;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import org.apache.pulsar.metadata.api.extended.SessionEvent;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKSessionWatcher
implements AutoCloseable,
Watcher {
    private static final Logger log = LoggerFactory.getLogger(ZKSessionWatcher.class);
    private final ZooKeeper zk;
    private SessionEvent currentStatus;
    private final Consumer<SessionEvent> sessionListener;
    private final long monitorTimeoutMillis;
    private final long tickTimeMillis;
    private final ScheduledExecutorService scheduler;
    private final ScheduledFuture<?> task;
    private long disconnectedAt = 0L;

    public ZKSessionWatcher(ZooKeeper zk, Consumer<SessionEvent> sessionListener) {
        this.zk = zk;
        this.monitorTimeoutMillis = zk.getSessionTimeout() * 5 / 6;
        this.tickTimeMillis = zk.getSessionTimeout() / 15;
        this.sessionListener = sessionListener;
        this.scheduler = Executors.newSingleThreadScheduledExecutor((ThreadFactory)new DefaultThreadFactory("metadata-store-zk-session-watcher"));
        this.task = this.scheduler.scheduleAtFixedRate(this::checkConnectionStatus, this.tickTimeMillis, this.tickTimeMillis, TimeUnit.MILLISECONDS);
        this.currentStatus = SessionEvent.SessionReestablished;
    }

    @Override
    public void close() throws Exception {
        this.task.cancel(true);
        this.scheduler.shutdownNow();
        this.scheduler.awaitTermination(10L, TimeUnit.SECONDS);
    }

    private synchronized void checkConnectionStatus() {
        try {
            Watcher.Event.KeeperState zkClientState;
            CompletableFuture future = new CompletableFuture();
            this.zk.exists("/", false, (rc, path, ctx, stat) -> {
                switch (KeeperException.Code.get((int)rc)) {
                    case CONNECTIONLOSS: {
                        future.complete(Watcher.Event.KeeperState.Disconnected);
                        break;
                    }
                    case SESSIONEXPIRED: {
                        future.complete(Watcher.Event.KeeperState.Expired);
                        break;
                    }
                    default: {
                        future.complete(Watcher.Event.KeeperState.SyncConnected);
                    }
                }
            }, null);
            try {
                zkClientState = (Watcher.Event.KeeperState)future.get(this.tickTimeMillis, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                zkClientState = Watcher.Event.KeeperState.Disconnected;
            }
            this.checkState(zkClientState);
        }
        catch (InterruptedException | RejectedExecutionException e) {
            this.task.cancel(true);
        }
        catch (Throwable t) {
            log.warn("Error while checking ZK connection status", t);
        }
    }

    public synchronized void process(WatchedEvent event) {
        log.info("Got ZK session watch event: {}", (Object)event);
        this.checkState(event.getState());
    }

    synchronized void setSessionInvalid() {
        this.currentStatus = SessionEvent.SessionLost;
    }

    private void checkState(Watcher.Event.KeeperState zkClientState) {
        switch (zkClientState) {
            case Expired: {
                if (this.currentStatus == SessionEvent.SessionLost) break;
                log.error("ZooKeeper session expired");
                this.currentStatus = SessionEvent.SessionLost;
                this.sessionListener.accept(this.currentStatus);
                break;
            }
            case Disconnected: {
                long timeRemainingMillis;
                if (this.disconnectedAt == 0L) {
                    this.disconnectedAt = System.nanoTime();
                }
                if ((timeRemainingMillis = this.monitorTimeoutMillis - TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.disconnectedAt)) <= 0L && this.currentStatus != SessionEvent.SessionLost) {
                    log.error("ZooKeeper session reconnection timeout. Notifying session is lost.");
                    this.currentStatus = SessionEvent.SessionLost;
                    this.sessionListener.accept(this.currentStatus);
                    break;
                }
                if (this.currentStatus == SessionEvent.SessionLost) break;
                log.warn("ZooKeeper client is disconnected. Waiting to reconnect, time remaining = {} seconds", (Object)((double)timeRemainingMillis / 1000.0));
                if (this.currentStatus != SessionEvent.SessionReestablished) break;
                this.currentStatus = SessionEvent.ConnectionLost;
                this.sessionListener.accept(this.currentStatus);
                break;
            }
            default: {
                if (this.currentStatus == SessionEvent.SessionReestablished) break;
                log.info("ZooKeeper client reconnection with server quorum. Current status: {}", (Object)this.currentStatus);
                this.disconnectedAt = 0L;
                this.sessionListener.accept(SessionEvent.Reconnected);
                if (this.currentStatus == SessionEvent.SessionLost) {
                    this.sessionListener.accept(SessionEvent.SessionReestablished);
                }
                this.currentStatus = SessionEvent.SessionReestablished;
            }
        }
    }
}

