/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.metadata.impl;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermission;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.metadata.api.GetResult;
import org.apache.pulsar.metadata.api.MetadataEventSynchronizer;
import org.apache.pulsar.metadata.api.MetadataStoreConfig;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.metadata.api.Notification;
import org.apache.pulsar.metadata.api.NotificationType;
import org.apache.pulsar.metadata.api.Stat;
import org.apache.pulsar.metadata.api.extended.CreateOption;
import org.apache.pulsar.metadata.impl.AbstractMetadataStore;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ConfigOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.InfoLogLevel;
import org.rocksdb.Options;
import org.rocksdb.OptionsUtil;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Transaction;
import org.rocksdb.TransactionDB;
import org.rocksdb.TransactionDBOptions;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RocksdbMetadataStore
extends AbstractMetadataStore {
    private static final Logger log = LoggerFactory.getLogger(RocksdbMetadataStore.class);
    static final String ROCKSDB_SCHEME = "rocksdb";
    static final String ROCKSDB_SCHEME_IDENTIFIER = "rocksdb:";
    private static final byte[] SEQUENTIAL_ID_KEY = RocksdbMetadataStore.toBytes("__metadata_sequentialId_key");
    private static final byte[] INSTANCE_ID_KEY = RocksdbMetadataStore.toBytes("__metadata_instanceId_key");
    private final long instanceId;
    private final AtomicLong sequentialIdGenerator;
    private final TransactionDB db;
    private final ReentrantReadWriteLock dbStateLock;
    private volatile State state;
    private final WriteOptions optionSync;
    private final WriteOptions optionDontSync;
    private final ReadOptions optionCache;
    private final ReadOptions optionDontCache;
    private MetadataEventSynchronizer synchronizer;
    private int referenceCount = 1;
    private static final Map<String, RocksdbMetadataStore> instancesCache = new ConcurrentHashMap<String, RocksdbMetadataStore>();
    private final String metadataUrl;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static RocksdbMetadataStore get(String metadataStoreUri, MetadataStoreConfig conf) throws MetadataStoreException {
        RocksdbMetadataStore store = instancesCache.get(metadataStoreUri);
        if (store != null) {
            RocksdbMetadataStore rocksdbMetadataStore = store;
            synchronized (rocksdbMetadataStore) {
                if (store.referenceCount > 0) {
                    ++store.referenceCount;
                    return store;
                }
            }
        }
        store = new RocksdbMetadataStore(metadataStoreUri, conf);
        store.synchronizer = conf.getSynchronizer();
        store.registerSyncLister(Optional.ofNullable(store.synchronizer));
        instancesCache.put(metadataStoreUri, store);
        return store;
    }

    @VisibleForTesting
    static byte[] toBytes(String s) {
        return s.getBytes(StandardCharsets.UTF_8);
    }

    @VisibleForTesting
    static String toString(byte[] bytes) {
        return new String(bytes, StandardCharsets.UTF_8);
    }

    @VisibleForTesting
    static byte[] toBytes(long value) {
        return ByteBuffer.wrap(new byte[8]).putLong(value).array();
    }

    @VisibleForTesting
    static long toLong(byte[] bytes) {
        return ByteBuffer.wrap(bytes).getLong();
    }

    private RocksdbMetadataStore(String metadataURL, MetadataStoreConfig metadataStoreConfig) throws MetadataStoreException {
        this.metadataUrl = metadataURL;
        try {
            RocksDB.loadLibrary();
        }
        catch (Throwable t) {
            throw new MetadataStoreException("Failed to load RocksDB JNI library", t);
        }
        String dataDir = metadataURL.substring(ROCKSDB_SCHEME_IDENTIFIER.length());
        Path dataPath = FileSystems.getDefault().getPath(dataDir, new String[0]);
        try {
            Files.createDirectories(dataPath, new FileAttribute[0]);
            Set<PosixFilePermission> perms = PosixFilePermissions.fromString("rwxr-x---");
            Files.setPosixFilePermissions(dataPath, perms);
        }
        catch (IOException e) {
            throw new MetadataStoreException("Fail to create RocksDB file directory", e);
        }
        this.db = this.openDB(dataPath.toString(), metadataStoreConfig.getConfigFilePath());
        this.optionSync = new WriteOptions().setSync(true);
        this.optionDontSync = new WriteOptions().setSync(false);
        this.optionCache = new ReadOptions().setFillCache(true);
        this.optionDontCache = new ReadOptions().setFillCache(false);
        try {
            this.sequentialIdGenerator = this.loadSequentialIdGenerator();
            this.instanceId = this.loadInstanceId();
        }
        catch (RocksDBException exception) {
            log.error("Error while init metastore state", (Throwable)exception);
            this.close();
            throw new MetadataStoreException("Error init metastore state", exception);
        }
        this.state = State.RUNNING;
        this.dbStateLock = new ReentrantReadWriteLock();
        log.info("new RocksdbMetadataStore,url={},instanceId={}", (Object)metadataStoreConfig, (Object)this.instanceId);
    }

    private long loadInstanceId() throws RocksDBException {
        byte[] value = this.db.get(this.optionDontCache, INSTANCE_ID_KEY);
        long instanceId = value != null ? RocksdbMetadataStore.toLong(value) + 1L : 0L;
        this.db.put(this.optionSync, INSTANCE_ID_KEY, RocksdbMetadataStore.toBytes(instanceId));
        return instanceId;
    }

    private AtomicLong loadSequentialIdGenerator() throws RocksDBException {
        AtomicLong generator = new AtomicLong(0L);
        byte[] value = this.db.get(this.optionDontCache, SEQUENTIAL_ID_KEY);
        if (value != null) {
            generator.set(RocksdbMetadataStore.toLong(value));
        } else {
            this.db.put(this.optionSync, INSTANCE_ID_KEY, RocksdbMetadataStore.toBytes(generator.get()));
        }
        return generator;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private TransactionDB openDB(String dataPath, String configFilePath) throws MetadataStoreException {
        try {
            TransactionDB transactionDB;
            TransactionDBOptions transactionDBOptions = new TransactionDBOptions();
            if (configFilePath != null) {
                DBOptions dbOptions = new DBOptions();
                ConfigOptions configOptions = new ConfigOptions();
                ArrayList cfDescriptors = new ArrayList();
                OptionsUtil.loadOptionsFromFile((ConfigOptions)configOptions, (String)configFilePath, (DBOptions)dbOptions, cfDescriptors);
                log.info("Load options from configFile({}), CF.size={},dbConfig={}", new Object[]{configFilePath, cfDescriptors.size(), dbOptions});
                if (log.isDebugEnabled()) {
                    for (ColumnFamilyDescriptor cfDescriptor : cfDescriptors) {
                        log.debug("CF={},Options={}", (Object)cfDescriptor.getName(), (Object)cfDescriptor.getOptions().toString());
                    }
                }
                ArrayList cfHandles = new ArrayList();
                try {
                    TransactionDB transactionDB2 = TransactionDB.open((DBOptions)dbOptions, (TransactionDBOptions)transactionDBOptions, (String)dataPath, cfDescriptors, cfHandles);
                    return transactionDB2;
                }
                finally {
                    dbOptions.close();
                    configOptions.close();
                }
            }
            Options options = new Options();
            options.setCreateIfMissing(true);
            this.configLog(options);
            try {
                transactionDB = TransactionDB.open((Options)options, (TransactionDBOptions)transactionDBOptions, (String)dataPath);
            }
            catch (Throwable throwable) {
                options.close();
                throw throwable;
            }
            options.close();
            return transactionDB;
            finally {
                transactionDBOptions.close();
            }
        }
        catch (RocksDBException e) {
            throw new MetadataStoreException("Error open RocksDB database", e);
        }
        catch (Throwable t) {
            throw MetadataStoreException.wrap(t);
        }
    }

    private void configLog(Options options) throws IOException {
        String logLevel;
        String logPath = System.getProperty("pulsar.log.dir", "");
        if (!logPath.isEmpty()) {
            Path logPathSetting = FileSystems.getDefault().getPath(logPath + "/rocksdb-log", new String[0]);
            Files.createDirectories(logPathSetting, new FileAttribute[0]);
            options.setDbLogDir(logPathSetting.toString());
            Set<PosixFilePermission> perms = PosixFilePermissions.fromString("rwxr-x---");
            Files.setPosixFilePermissions(logPathSetting, perms);
        }
        switch (logLevel = System.getProperty("pulsar.log.level", "info")) {
            case "debug": {
                options.setInfoLogLevel(InfoLogLevel.DEBUG_LEVEL);
                break;
            }
            case "info": {
                options.setInfoLogLevel(InfoLogLevel.INFO_LEVEL);
                break;
            }
            case "warn": {
                options.setInfoLogLevel(InfoLogLevel.WARN_LEVEL);
                break;
            }
            case "error": {
                options.setInfoLogLevel(InfoLogLevel.ERROR_LEVEL);
                break;
            }
            default: {
                log.warn("Unrecognized RockDB log level: {}", (Object)logLevel);
            }
        }
        options.setKeepLogFileNum(30L);
        options.setLogFileTimeToRoll(TimeUnit.DAYS.toSeconds(1L));
    }

    @Override
    public synchronized void close() throws MetadataStoreException {
        --this.referenceCount;
        if (this.referenceCount > 0) {
            return;
        }
        instancesCache.remove(this.metadataUrl, this);
        if (this.state == State.CLOSED) {
            return;
        }
        try {
            this.dbStateLock.writeLock().lock();
            this.state = State.CLOSED;
            log.info("close.instanceId={}", (Object)this.instanceId);
            this.db.close();
            this.optionSync.close();
            this.optionDontSync.close();
            this.optionCache.close();
            this.optionDontCache.close();
            super.close();
        }
        catch (Throwable throwable) {
            throw MetadataStoreException.wrap(throwable);
        }
        finally {
            this.dbStateLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CompletableFuture<Optional<GetResult>> storeGet(String path) {
        if (log.isDebugEnabled()) {
            log.debug("getFromStore.path={},instanceId={}", (Object)path, (Object)this.instanceId);
        }
        try {
            this.dbStateLock.readLock().lock();
            if (this.state == State.CLOSED) {
                throw new MetadataStoreException.AlreadyClosedException("");
            }
            byte[] value = this.db.get(this.optionCache, RocksdbMetadataStore.toBytes(path));
            if (value == null) {
                CompletableFuture<Optional<GetResult>> completableFuture = CompletableFuture.completedFuture(Optional.empty());
                return completableFuture;
            }
            MetaValue metaValue = MetaValue.parse(value);
            if (metaValue.ephemeral && metaValue.owner != this.instanceId) {
                this.delete(path, Optional.empty());
                CompletableFuture<Optional<GetResult>> completableFuture = CompletableFuture.completedFuture(Optional.empty());
                return completableFuture;
            }
            GetResult result = new GetResult(metaValue.getData(), new Stat(path, metaValue.getVersion(), metaValue.getCreatedTimestamp(), metaValue.getModifiedTimestamp(), metaValue.ephemeral, metaValue.getOwner() == this.instanceId));
            CompletableFuture<Optional<GetResult>> completableFuture = CompletableFuture.completedFuture(Optional.of(result));
            return completableFuture;
        }
        catch (Throwable e) {
            CompletableFuture completableFuture = FutureUtil.failedFuture((Throwable)MetadataStoreException.wrap(e));
            return completableFuture;
        }
        finally {
            this.dbStateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected CompletableFuture<List<String>> getChildrenFromStore(String path) {
        if (log.isDebugEnabled()) {
            log.debug("getChildrenFromStore.path={},instanceId={}", (Object)path, (Object)this.instanceId);
        }
        try {
            CompletableFuture<List<String>> completableFuture;
            block17: {
                this.dbStateLock.readLock().lock();
                if (this.state == State.CLOSED) {
                    throw new MetadataStoreException.AlreadyClosedException("");
                }
                RocksIterator iterator = this.db.newIterator(this.optionDontCache);
                try {
                    String currentPath;
                    HashSet<String> result = new HashSet<String>();
                    Object firstKey = path.equals("/") ? path : path + "/";
                    Object lastKey = path.equals("/") ? "0" : path + "0";
                    iterator.seek(RocksdbMetadataStore.toBytes((String)firstKey));
                    while (iterator.isValid() && ((String)lastKey).compareTo(currentPath = RocksdbMetadataStore.toString(iterator.key())) > 0) {
                        byte[] value = iterator.value();
                        if (value != null) {
                            MetaValue metaValue = MetaValue.parse(value);
                            if (metaValue.ephemeral && metaValue.owner != this.instanceId) {
                                this.delete(currentPath, Optional.empty());
                            } else {
                                String relativePath = currentPath.substring(((String)firstKey).length());
                                String child = relativePath.split("/", 2)[0];
                                result.add(child);
                            }
                        }
                        iterator.next();
                    }
                    ArrayList arrayResult = new ArrayList(result);
                    arrayResult.sort(Comparator.naturalOrder());
                    completableFuture = CompletableFuture.completedFuture(arrayResult);
                    if (iterator == null) break block17;
                }
                catch (Throwable throwable) {
                    try {
                        if (iterator != null) {
                            try {
                                iterator.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Throwable e) {
                        CompletableFuture completableFuture2 = FutureUtil.failedFuture((Throwable)MetadataStoreException.wrap(e));
                        return completableFuture2;
                    }
                }
                iterator.close();
            }
            return completableFuture;
        }
        finally {
            this.dbStateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected CompletableFuture<Boolean> existsFromStore(String path) {
        if (log.isDebugEnabled()) {
            log.debug("existsFromStore.path={},instanceId={}", (Object)path, (Object)this.instanceId);
        }
        try {
            this.dbStateLock.readLock().lock();
            if (this.state == State.CLOSED) {
                throw new MetadataStoreException.AlreadyClosedException("");
            }
            byte[] value = this.db.get(this.optionDontCache, RocksdbMetadataStore.toBytes(path));
            if (log.isDebugEnabled() && value != null) {
                MetaValue metaValue = MetaValue.parse(value);
                log.debug("Get data from db:{}.", (Object)metaValue);
            }
            CompletableFuture<Boolean> completableFuture = CompletableFuture.completedFuture(value != null);
            return completableFuture;
        }
        catch (Throwable e) {
            CompletableFuture completableFuture = FutureUtil.failedFuture((Throwable)MetadataStoreException.wrap(e));
            return completableFuture;
        }
        finally {
            this.dbStateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected CompletableFuture<Void> storeDelete(String path, Optional<Long> expectedVersion) {
        if (log.isDebugEnabled()) {
            log.debug("storeDelete.path={},instanceId={}", (Object)path, (Object)this.instanceId);
        }
        try {
            CompletableFuture<Object> completableFuture;
            block16: {
                this.dbStateLock.readLock().lock();
                if (this.state == State.CLOSED) {
                    throw new MetadataStoreException.AlreadyClosedException("");
                }
                Transaction transaction = this.db.beginTransaction(this.optionSync);
                try {
                    byte[] pathBytes = RocksdbMetadataStore.toBytes(path);
                    byte[] oldValueData = transaction.getForUpdate(this.optionDontCache, pathBytes, true);
                    MetaValue metaValue = MetaValue.parse(oldValueData);
                    if (metaValue == null) {
                        throw new MetadataStoreException.NotFoundException(String.format("path %s not found.", path));
                    }
                    if (expectedVersion.isPresent() && !expectedVersion.get().equals(metaValue.getVersion())) {
                        throw new MetadataStoreException.BadVersionException(String.format("Version mismatch, actual=%s, expect=%s", metaValue.getVersion(), expectedVersion.get()));
                    }
                    transaction.delete(pathBytes);
                    transaction.commit();
                    this.receivedNotification(new Notification(NotificationType.Deleted, path));
                    this.notifyParentChildrenChanged(path);
                    completableFuture = CompletableFuture.completedFuture(null);
                    if (transaction == null) break block16;
                }
                catch (Throwable throwable) {
                    try {
                        if (transaction != null) {
                            try {
                                transaction.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Throwable e) {
                        if (log.isDebugEnabled()) {
                            log.debug("error in storeDelete,path={}", (Object)path, (Object)e);
                        }
                        CompletableFuture completableFuture2 = FutureUtil.failedFuture((Throwable)MetadataStoreException.wrap(e));
                        return completableFuture2;
                    }
                }
                transaction.close();
            }
            return completableFuture;
        }
        finally {
            this.dbStateLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected CompletableFuture<Stat> storePut(String path, byte[] data, Optional<Long> expectedVersion, EnumSet<CreateOption> options) {
        if (log.isDebugEnabled()) {
            log.debug("storePut.path={},instanceId={}", path, (Object)this.instanceId);
        }
        try {
            CompletableFuture<Stat> completableFuture;
            block19: {
                this.dbStateLock.readLock().lock();
                if (this.state == State.CLOSED) {
                    throw new MetadataStoreException.AlreadyClosedException("");
                }
                Transaction transaction = this.db.beginTransaction(this.optionSync);
                try {
                    byte[] pathBytes = RocksdbMetadataStore.toBytes((String)path);
                    byte[] oldValueData = transaction.getForUpdate(this.optionDontCache, pathBytes, true);
                    MetaValue metaValue = MetaValue.parse(oldValueData);
                    if (expectedVersion.isPresent() && (metaValue == null && expectedVersion.get() != -1L || metaValue != null && !expectedVersion.get().equals(metaValue.getVersion()))) {
                        throw new MetadataStoreException.BadVersionException(String.format("Version mismatch, actual=%s, expect=%s", metaValue == null ? null : Long.valueOf(metaValue.getVersion()), expectedVersion.get()));
                    }
                    boolean created = false;
                    long timestamp = System.currentTimeMillis();
                    if (metaValue == null) {
                        metaValue = new MetaValue();
                        metaValue.version = 0L;
                        metaValue.createdTimestamp = timestamp;
                        metaValue.ephemeral = options.contains((Object)CreateOption.Ephemeral);
                        if (options.contains((Object)CreateOption.Sequential)) {
                            path = (String)path + this.sequentialIdGenerator.getAndIncrement();
                            pathBytes = RocksdbMetadataStore.toBytes((String)path);
                            transaction.put(SEQUENTIAL_ID_KEY, RocksdbMetadataStore.toBytes(this.sequentialIdGenerator.get()));
                        }
                        created = true;
                    } else {
                        ++metaValue.version;
                    }
                    metaValue.modifiedTimestamp = timestamp;
                    metaValue.owner = this.instanceId;
                    metaValue.data = data;
                    transaction.put(pathBytes, metaValue.serialize());
                    transaction.commit();
                    this.receivedNotification(new Notification(created ? NotificationType.Created : NotificationType.Modified, (String)path));
                    if (created) {
                        this.notifyParentChildrenChanged((String)path);
                    }
                    completableFuture = CompletableFuture.completedFuture(new Stat((String)path, metaValue.version, metaValue.createdTimestamp, metaValue.modifiedTimestamp, metaValue.ephemeral, true));
                    if (transaction == null) break block19;
                }
                catch (Throwable throwable) {
                    try {
                        if (transaction != null) {
                            try {
                                transaction.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (Throwable e) {
                        if (log.isDebugEnabled()) {
                            log.debug("error in storePut,path={}", path, (Object)e);
                        }
                        CompletableFuture completableFuture2 = FutureUtil.failedFuture((Throwable)MetadataStoreException.wrap(e));
                        return completableFuture2;
                    }
                }
                transaction.close();
            }
            return completableFuture;
        }
        finally {
            this.dbStateLock.readLock().unlock();
        }
    }

    @Override
    public Optional<MetadataEventSynchronizer> getMetadataEventSynchronizer() {
        return Optional.ofNullable(this.synchronizer);
    }

    static enum State {
        RUNNING,
        CLOSED;

    }

    static class MetaValue {
        private static final int HEADER_SIZE = 41;
        private static final int FORMAT_VERSION_V1 = 1;
        long version;
        long owner;
        long createdTimestamp;
        long modifiedTimestamp;
        boolean ephemeral;
        byte[] data;

        public byte[] serialize() {
            byte[] result = new byte[41 + this.data.length];
            ByteBuffer buffer = ByteBuffer.wrap(result);
            buffer.putInt(41);
            buffer.putInt(1);
            buffer.putLong(this.version);
            buffer.putLong(this.owner);
            buffer.putLong(this.createdTimestamp);
            buffer.putLong(this.modifiedTimestamp);
            buffer.put((byte)(this.ephemeral ? 1 : 0));
            buffer.put(this.data);
            return result;
        }

        public static MetaValue parse(byte[] dataBytes) throws MetadataStoreException {
            if (dataBytes == null) {
                return null;
            }
            if (dataBytes.length < 4) {
                throw new MetadataStoreException("Invalid MetaValue data, size=" + dataBytes.length);
            }
            ByteBuffer buffer = ByteBuffer.wrap(dataBytes);
            MetaValue metaValue = new MetaValue();
            int headerSize = buffer.getInt();
            if (dataBytes.length < headerSize) {
                throw new MetadataStoreException(String.format("Invalid MetaValue data, no enough header data. expect %d, actual %d", headerSize, dataBytes.length));
            }
            int formatVersion = buffer.getInt();
            if (formatVersion < 1) {
                throw new MetadataStoreException("Invalid MetaValue format version=" + formatVersion);
            }
            metaValue.version = buffer.getLong();
            metaValue.owner = buffer.getLong();
            metaValue.createdTimestamp = buffer.getLong();
            metaValue.modifiedTimestamp = buffer.getLong();
            metaValue.ephemeral = buffer.get() > 0;
            buffer.position(headerSize);
            metaValue.data = new byte[buffer.remaining()];
            buffer.get(metaValue.data);
            return metaValue;
        }

        public long getVersion() {
            return this.version;
        }

        public long getOwner() {
            return this.owner;
        }

        public long getCreatedTimestamp() {
            return this.createdTimestamp;
        }

        public long getModifiedTimestamp() {
            return this.modifiedTimestamp;
        }

        public boolean isEphemeral() {
            return this.ephemeral;
        }

        public byte[] getData() {
            return this.data;
        }

        public void setVersion(long version) {
            this.version = version;
        }

        public void setOwner(long owner) {
            this.owner = owner;
        }

        public void setCreatedTimestamp(long createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        public void setModifiedTimestamp(long modifiedTimestamp) {
            this.modifiedTimestamp = modifiedTimestamp;
        }

        public void setEphemeral(boolean ephemeral) {
            this.ephemeral = ephemeral;
        }

        public void setData(byte[] data) {
            this.data = data;
        }

        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }
            if (!(o instanceof MetaValue)) {
                return false;
            }
            MetaValue other = (MetaValue)o;
            if (!other.canEqual(this)) {
                return false;
            }
            if (this.getVersion() != other.getVersion()) {
                return false;
            }
            if (this.getOwner() != other.getOwner()) {
                return false;
            }
            if (this.getCreatedTimestamp() != other.getCreatedTimestamp()) {
                return false;
            }
            if (this.getModifiedTimestamp() != other.getModifiedTimestamp()) {
                return false;
            }
            if (this.isEphemeral() != other.isEphemeral()) {
                return false;
            }
            return Arrays.equals(this.getData(), other.getData());
        }

        protected boolean canEqual(Object other) {
            return other instanceof MetaValue;
        }

        public int hashCode() {
            int PRIME = 59;
            int result = 1;
            long $version = this.getVersion();
            result = result * 59 + (int)($version >>> 32 ^ $version);
            long $owner = this.getOwner();
            result = result * 59 + (int)($owner >>> 32 ^ $owner);
            long $createdTimestamp = this.getCreatedTimestamp();
            result = result * 59 + (int)($createdTimestamp >>> 32 ^ $createdTimestamp);
            long $modifiedTimestamp = this.getModifiedTimestamp();
            result = result * 59 + (int)($modifiedTimestamp >>> 32 ^ $modifiedTimestamp);
            result = result * 59 + (this.isEphemeral() ? 79 : 97);
            result = result * 59 + Arrays.hashCode(this.getData());
            return result;
        }

        public String toString() {
            return "RocksdbMetadataStore.MetaValue(version=" + this.getVersion() + ", owner=" + this.getOwner() + ", createdTimestamp=" + this.getCreatedTimestamp() + ", modifiedTimestamp=" + this.getModifiedTimestamp() + ", ephemeral=" + this.isEphemeral() + ", data=" + Arrays.toString(this.getData()) + ")";
        }

        public MetaValue(long version, long owner, long createdTimestamp, long modifiedTimestamp, boolean ephemeral, byte[] data) {
            this.version = version;
            this.owner = owner;
            this.createdTimestamp = createdTimestamp;
            this.modifiedTimestamp = modifiedTimestamp;
            this.ephemeral = ephemeral;
            this.data = data;
        }

        public MetaValue() {
        }
    }
}

