/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.metadata.impl;

import com.google.common.annotations.VisibleForTesting;
import java.io.File;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.bookkeeper.zookeeper.BoundExponentialBackoffRetryPolicy;
import org.apache.bookkeeper.zookeeper.RetryPolicy;
import org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.metadata.api.GetResult;
import org.apache.pulsar.metadata.api.MetadataStoreConfig;
import org.apache.pulsar.metadata.api.MetadataStoreException;
import org.apache.pulsar.metadata.api.MetadataStoreLifecycle;
import org.apache.pulsar.metadata.api.Notification;
import org.apache.pulsar.metadata.api.NotificationType;
import org.apache.pulsar.metadata.api.Stat;
import org.apache.pulsar.metadata.api.extended.CreateOption;
import org.apache.pulsar.metadata.api.extended.MetadataStoreExtended;
import org.apache.pulsar.metadata.api.extended.SessionEvent;
import org.apache.pulsar.metadata.impl.PulsarZooKeeperClient;
import org.apache.pulsar.metadata.impl.ZKSessionWatcher;
import org.apache.pulsar.metadata.impl.batching.AbstractBatchedMetadataStore;
import org.apache.pulsar.metadata.impl.batching.MetadataOp;
import org.apache.pulsar.metadata.impl.batching.OpDelete;
import org.apache.pulsar.metadata.impl.batching.OpGet;
import org.apache.pulsar.metadata.impl.batching.OpGetChildren;
import org.apache.pulsar.metadata.impl.batching.OpPut;
import org.apache.zookeeper.AddWatchMode;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Op;
import org.apache.zookeeper.OpResult;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.client.ConnectStringParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKMetadataStore
extends AbstractBatchedMetadataStore
implements MetadataStoreExtended,
MetadataStoreLifecycle {
    private static final Logger log = LoggerFactory.getLogger(ZKMetadataStore.class);
    public static final String ZK_SCHEME = "zk";
    public static final String ZK_SCHEME_IDENTIFIER = "zk:";
    private final String zkConnectString;
    private final String rootPath;
    private final MetadataStoreConfig metadataStoreConfig;
    private final boolean isZkManaged;
    private final ZooKeeper zkc;
    private Optional<ZKSessionWatcher> sessionWatcher;

    public ZKMetadataStore(String metadataURL, MetadataStoreConfig metadataStoreConfig, boolean enableSessionWatcher) throws MetadataStoreException {
        super(metadataStoreConfig);
        try {
            this.zkConnectString = metadataURL.startsWith(ZK_SCHEME_IDENTIFIER) ? metadataURL.substring(ZK_SCHEME_IDENTIFIER.length()) : metadataURL;
            this.metadataStoreConfig = metadataStoreConfig;
            this.rootPath = new ConnectStringParser(this.zkConnectString).getChrootPath();
            this.isZkManaged = true;
            this.zkc = PulsarZooKeeperClient.newBuilder().connectString(this.zkConnectString).connectRetryPolicy((RetryPolicy)new BoundExponentialBackoffRetryPolicy(100L, 60000L, Integer.MAX_VALUE)).allowReadOnlyMode(metadataStoreConfig.isAllowReadOnlyOperations()).sessionTimeoutMs(metadataStoreConfig.getSessionTimeoutMillis()).watchers(Collections.singleton(event -> {
                if (this.sessionWatcher != null) {
                    this.sessionWatcher.ifPresent(sw -> this.executor.execute(() -> sw.process(event)));
                }
            })).build();
            this.zkc.addWatch("/", this::handleWatchEvent, AddWatchMode.PERSISTENT_RECURSIVE);
            this.sessionWatcher = enableSessionWatcher ? Optional.of(new ZKSessionWatcher(this.zkc, this::receivedSessionEvent)) : Optional.empty();
        }
        catch (Throwable t) {
            throw new MetadataStoreException(t);
        }
    }

    @VisibleForTesting
    public ZKMetadataStore(ZooKeeper zkc) {
        this(zkc, MetadataStoreConfig.builder().build());
    }

    @VisibleForTesting
    public ZKMetadataStore(ZooKeeper zkc, MetadataStoreConfig config) {
        super(config);
        this.zkConnectString = null;
        this.rootPath = null;
        this.metadataStoreConfig = null;
        this.isZkManaged = false;
        this.zkc = zkc;
        this.sessionWatcher = Optional.of(new ZKSessionWatcher(zkc, this::receivedSessionEvent));
        zkc.addWatch("/", this::handleWatchEvent, AddWatchMode.PERSISTENT_RECURSIVE);
    }

    @Override
    protected void receivedSessionEvent(SessionEvent event) {
        if (event == SessionEvent.SessionReestablished) {
            this.zkc.addWatch("/", this::handleWatchEvent, AddWatchMode.PERSISTENT_RECURSIVE, (rc, path, ctx) -> {
                if (rc == KeeperException.Code.OK.intValue()) {
                    super.receivedSessionEvent(event);
                } else {
                    log.error("Failed to recreate persistent watch on ZooKeeper: {}", (Object)KeeperException.Code.get((int)rc));
                    this.sessionWatcher.ifPresent(ZKSessionWatcher::setSessionInvalid);
                    if (this.zkc instanceof PulsarZooKeeperClient) {
                        ((PulsarZooKeeperClient)this.zkc).process(new WatchedEvent(Watcher.Event.EventType.None, Watcher.Event.KeeperState.Expired, null));
                    }
                }
            }, null);
        } else {
            super.receivedSessionEvent(event);
        }
    }

    @Override
    public CompletableFuture<Void> sync(final String path) {
        final CompletableFuture<Void> result = new CompletableFuture<Void>();
        this.zkc.sync(path, new AsyncCallback.VoidCallback(){

            public void processResult(int rc, String s, Object o) {
                KeeperException.Code code = KeeperException.Code.get((int)rc);
                if (code == KeeperException.Code.OK) {
                    result.complete(null);
                } else {
                    MetadataStoreException e = ZKMetadataStore.getException(code, path);
                    result.completeExceptionally(e);
                }
            }
        }, null);
        return result;
    }

    @Override
    protected void batchOperation(List<MetadataOp> ops) {
        try {
            this.zkc.multi((Iterable)ops.stream().map(this::convertOp).collect(Collectors.toList()), (rc, path, ctx, results) -> {
                if (results == null) {
                    KeeperException.Code code = KeeperException.Code.get((int)rc);
                    if (code == KeeperException.Code.CONNECTIONLOSS) {
                        String countsByType = ops.stream().collect(Collectors.groupingBy(MetadataOp::getType, Collectors.summingInt(op -> 1))).entrySet().stream().map(e -> e.getValue() + " " + ((MetadataOp.Type)((Object)((Object)((Object)e.getKey())))).name() + " entries").collect(Collectors.joining(", "));
                        Long totalSize = ops.stream().collect(Collectors.summingLong(MetadataOp::size));
                        log.warn("Connection loss while executing batch operation of {} of total data size of {}. Retrying individual operations one-by-one.", (Object)countsByType, (Object)totalSize);
                        this.executor.schedule(() -> ops.forEach(o -> this.batchOperation(Collections.singletonList(o))), 100L, TimeUnit.MILLISECONDS);
                    } else {
                        MetadataStoreException e2 = ZKMetadataStore.getException(code, path);
                        ops.forEach(o -> o.getFuture().completeExceptionally(e2));
                    }
                    return;
                }
                this.execute(() -> {
                    block6: for (int i = 0; i < ops.size(); ++i) {
                        OpResult opr = (OpResult)results.get(i);
                        MetadataOp op = (MetadataOp)ops.get(i);
                        switch (op.getType()) {
                            case PUT: {
                                this.handlePutResult(op.asPut(), opr);
                                continue block6;
                            }
                            case DELETE: {
                                this.handleDeleteResult(op.asDelete(), opr);
                                continue block6;
                            }
                            case GET: {
                                this.handleGetResult(op.asGet(), opr);
                                continue block6;
                            }
                            case GET_CHILDREN: {
                                this.handleGetChildrenResult(op.asGetChildren(), opr);
                                continue block6;
                            }
                            default: {
                                op.getFuture().completeExceptionally(new MetadataStoreException("Operation type not supported in multi: " + op.getType()));
                            }
                        }
                    }
                }, () -> ops.stream().map(MetadataOp::getFuture).collect(Collectors.toList()));
            }, null);
        }
        catch (Throwable t) {
            ops.forEach(o -> o.getFuture().completeExceptionally(new MetadataStoreException(t)));
        }
    }

    private void handlePutResult(OpPut op, OpResult opr) {
        if (opr instanceof OpResult.ErrorResult) {
            OpResult.ErrorResult er = (OpResult.ErrorResult)opr;
            KeeperException.Code code = KeeperException.Code.get((int)er.getErr());
            if (code == KeeperException.Code.NONODE) {
                this.internalStorePut(op);
            } else if (code == KeeperException.Code.NODEEXISTS) {
                op.getFuture().completeExceptionally(ZKMetadataStore.getException(KeeperException.Code.BADVERSION, op.getPath()));
            } else if (code == KeeperException.Code.RUNTIMEINCONSISTENCY || code == KeeperException.Code.OK) {
                this.internalStorePut(op);
            } else {
                op.getFuture().completeExceptionally(ZKMetadataStore.getException(code, op.getPath()));
            }
        } else if (opr instanceof OpResult.CreateResult) {
            OpResult.CreateResult cr = (OpResult.CreateResult)opr;
            op.getFuture().complete(new Stat(cr.getPath(), 0L, 0L, 0L, op.isEphemeral(), true));
        } else {
            OpResult.SetDataResult sdr = (OpResult.SetDataResult)opr;
            op.getFuture().complete(this.getStat(op.getPath(), sdr.getStat()));
        }
    }

    private void handleGetResult(OpGet op, OpResult opr) {
        if (opr instanceof OpResult.ErrorResult) {
            OpResult.ErrorResult er = (OpResult.ErrorResult)opr;
            KeeperException.Code code = KeeperException.Code.get((int)er.getErr());
            if (code == KeeperException.Code.NONODE) {
                op.getFuture().complete(Optional.empty());
            } else {
                op.getFuture().completeExceptionally(ZKMetadataStore.getException(code, op.getPath()));
            }
        } else {
            OpResult.GetDataResult gdr = (OpResult.GetDataResult)opr;
            op.getFuture().complete(Optional.of(new GetResult(gdr.getData(), this.getStat(op.getPath(), gdr.getStat()))));
        }
    }

    private void handleGetChildrenResult(OpGetChildren op, OpResult opr) {
        if (opr instanceof OpResult.ErrorResult) {
            OpResult.ErrorResult er = (OpResult.ErrorResult)opr;
            KeeperException.Code code = KeeperException.Code.get((int)er.getErr());
            if (code == KeeperException.Code.NONODE) {
                op.asGetChildren().getFuture().complete(Collections.emptyList());
            } else {
                op.getFuture().completeExceptionally(ZKMetadataStore.getException(code, op.getPath()));
            }
        } else {
            OpResult.GetChildrenResult gdr = (OpResult.GetChildrenResult)opr;
            Collections.sort(gdr.getChildren());
            op.getFuture().complete(gdr.getChildren());
        }
    }

    private void handleDeleteResult(OpDelete op, OpResult opr) {
        if (opr instanceof OpResult.ErrorResult) {
            OpResult.ErrorResult er = (OpResult.ErrorResult)opr;
            KeeperException.Code code = KeeperException.Code.get((int)er.getErr());
            if (code == KeeperException.Code.RUNTIMEINCONSISTENCY || code == KeeperException.Code.OK) {
                this.internalStoreDelete(op);
            } else {
                op.getFuture().completeExceptionally(ZKMetadataStore.getException(code, op.getPath()));
            }
        } else {
            op.getFuture().complete(null);
        }
    }

    private Op convertOp(MetadataOp op) {
        switch (op.getType()) {
            case GET: {
                return Op.getData((String)op.asGet().getPath());
            }
            case PUT: {
                OpPut p = op.asPut();
                CreateMode createMode = ZKMetadataStore.getCreateMode(p.getOptions());
                if (p.getOptExpectedVersion().isPresent() && p.getOptExpectedVersion().get() == -1L) {
                    return Op.create((String)p.getPath(), (byte[])p.getData(), (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, (CreateMode)createMode);
                }
                return Op.setData((String)p.getPath(), (byte[])p.getData(), (int)p.getOptExpectedVersion().orElse(-1L).intValue());
            }
            case DELETE: {
                OpDelete d = op.asDelete();
                return Op.delete((String)d.getPath(), (int)d.getOptExpectedVersion().orElse(-1L).intValue());
            }
            case GET_CHILDREN: {
                return Op.getChildren((String)op.asGetChildren().getPath());
            }
        }
        return null;
    }

    @Override
    public CompletableFuture<Boolean> existsFromStore(String path) {
        CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
        try {
            this.zkc.exists(path, null, (rc, path1, ctx, stat) -> this.execute(() -> {
                KeeperException.Code code = KeeperException.Code.get((int)rc);
                if (code == KeeperException.Code.OK) {
                    future.complete(true);
                } else if (code == KeeperException.Code.NONODE) {
                    future.complete(false);
                } else {
                    future.completeExceptionally(ZKMetadataStore.getException(code, path));
                }
            }, future), future);
        }
        catch (Throwable t) {
            future.completeExceptionally(new MetadataStoreException(t));
        }
        return future;
    }

    private void internalStoreDelete(OpDelete op) {
        int expectedVersion = op.getOptExpectedVersion().orElse(-1L).intValue();
        CompletableFuture<Void> future = op.getFuture();
        try {
            this.zkc.delete(op.getPath(), expectedVersion, (rc, path1, ctx) -> this.execute(() -> {
                KeeperException.Code code = KeeperException.Code.get((int)rc);
                if (code == KeeperException.Code.OK) {
                    future.complete(null);
                } else {
                    future.completeExceptionally(ZKMetadataStore.getException(code, op.getPath()));
                }
            }, future), null);
        }
        catch (Throwable t) {
            future.completeExceptionally(new MetadataStoreException(t));
        }
    }

    private void internalStorePut(OpPut opPut) {
        boolean hasVersion = opPut.getOptExpectedVersion().isPresent();
        int expectedVersion = opPut.getOptExpectedVersion().orElse(-1L).intValue();
        CompletableFuture<Stat> future = opPut.getFuture();
        try {
            if (hasVersion && expectedVersion == -1) {
                CreateMode createMode = ZKMetadataStore.getCreateMode(opPut.getOptions());
                ZKMetadataStore.asyncCreateFullPathOptimistic(this.zkc, opPut.getPath(), opPut.getData(), createMode, (rc, path1, ctx, name) -> this.execute(() -> {
                    KeeperException.Code code = KeeperException.Code.get((int)rc);
                    if (code == KeeperException.Code.OK) {
                        future.complete(new Stat(name, 0L, 0L, 0L, createMode.isEphemeral(), true));
                    } else if (code == KeeperException.Code.NODEEXISTS) {
                        future.completeExceptionally(ZKMetadataStore.getException(KeeperException.Code.BADVERSION, opPut.getPath()));
                    } else {
                        future.completeExceptionally(ZKMetadataStore.getException(code, opPut.getPath()));
                    }
                }, future));
            } else {
                this.zkc.setData(opPut.getPath(), opPut.getData(), expectedVersion, (rc, path1, ctx, stat) -> this.execute(() -> {
                    KeeperException.Code code = KeeperException.Code.get((int)rc);
                    if (code == KeeperException.Code.OK) {
                        future.complete(this.getStat(path1, stat));
                    } else if (code == KeeperException.Code.NONODE) {
                        if (hasVersion) {
                            future.completeExceptionally(ZKMetadataStore.getException(KeeperException.Code.BADVERSION, opPut.getPath()));
                        } else {
                            ((CompletableFuture)this.put(opPut.getPath(), opPut.getData(), Optional.of(-1L)).thenAccept(s -> future.complete((Stat)s))).exceptionally(ex -> {
                                if (ex.getCause() instanceof MetadataStoreException.BadVersionException) {
                                    this.internalStorePut(opPut);
                                } else {
                                    future.completeExceptionally(MetadataStoreException.wrap(ex.getCause()));
                                }
                                return null;
                            });
                        }
                    } else {
                        future.completeExceptionally(ZKMetadataStore.getException(code, opPut.getPath()));
                    }
                }, future), null);
            }
        }
        catch (Throwable t) {
            future.completeExceptionally(new MetadataStoreException(t));
        }
    }

    @Override
    public void close() throws Exception {
        if (this.isZkManaged) {
            this.zkc.close();
        }
        if (this.sessionWatcher.isPresent()) {
            this.sessionWatcher.get().close();
        }
        super.close();
    }

    private Stat getStat(String path, org.apache.zookeeper.data.Stat zkStat) {
        return new Stat(path, zkStat.getVersion(), zkStat.getCtime(), zkStat.getMtime(), zkStat.getEphemeralOwner() != -1L, zkStat.getEphemeralOwner() == this.zkc.getSessionId());
    }

    private static MetadataStoreException getException(KeeperException.Code code, String path) {
        KeeperException ex = KeeperException.create((KeeperException.Code)code, (String)path);
        switch (code) {
            case BADVERSION: {
                return new MetadataStoreException.BadVersionException(ex);
            }
            case NONODE: {
                return new MetadataStoreException.NotFoundException(ex);
            }
            case NODEEXISTS: {
                return new MetadataStoreException.AlreadyExistsException(ex);
            }
        }
        return new MetadataStoreException(ex);
    }

    private void handleWatchEvent(WatchedEvent event) {
        NotificationType type;
        String path;
        if (log.isDebugEnabled()) {
            log.debug("Received ZK watch : {}", (Object)event);
        }
        if ((path = event.getPath()) == null) {
            return;
        }
        String parent = ZKMetadataStore.parent(path);
        Notification childrenChangedNotification = null;
        switch (event.getType()) {
            case NodeCreated: {
                type = NotificationType.Created;
                if (parent == null) break;
                childrenChangedNotification = new Notification(NotificationType.ChildrenChanged, parent);
                break;
            }
            case NodeDataChanged: {
                type = NotificationType.Modified;
                break;
            }
            case NodeChildrenChanged: {
                type = NotificationType.ChildrenChanged;
                break;
            }
            case NodeDeleted: {
                type = NotificationType.Deleted;
                if (parent == null) break;
                childrenChangedNotification = new Notification(NotificationType.ChildrenChanged, parent);
                break;
            }
            default: {
                return;
            }
        }
        this.receivedNotification(new Notification(type, event.getPath()));
        if (childrenChangedNotification != null) {
            this.receivedNotification(childrenChangedNotification);
        }
    }

    private static CreateMode getCreateMode(EnumSet<CreateOption> options) {
        if (options.contains((Object)CreateOption.Ephemeral)) {
            if (options.contains((Object)CreateOption.Sequential)) {
                return CreateMode.EPHEMERAL_SEQUENTIAL;
            }
            return CreateMode.EPHEMERAL;
        }
        if (options.contains((Object)CreateOption.Sequential)) {
            return CreateMode.PERSISTENT_SEQUENTIAL;
        }
        return CreateMode.PERSISTENT;
    }

    public long getZkSessionId() {
        return this.zkc.getSessionId();
    }

    public ZooKeeper getZkClient() {
        return this.zkc;
    }

    @Override
    public CompletableFuture<Void> initializeCluster() {
        if (this.zkConnectString == null) {
            return FutureUtil.failedFuture((Throwable)new MetadataStoreException("metadataURL is not set"));
        }
        if (this.metadataStoreConfig == null) {
            return FutureUtil.failedFuture((Throwable)new MetadataStoreException("metadataStoreConfig is not set"));
        }
        int chrootIndex = this.zkConnectString.indexOf("/");
        if (chrootIndex > 0) {
            String chrootPath = this.zkConnectString.substring(chrootIndex);
            String zkConnectForChrootCreation = this.zkConnectString.substring(0, chrootIndex);
            try (PulsarZooKeeperClient chrootZk = PulsarZooKeeperClient.newBuilder().connectString(zkConnectForChrootCreation).sessionTimeoutMs(this.metadataStoreConfig.getSessionTimeoutMillis()).connectRetryPolicy((RetryPolicy)new BoundExponentialBackoffRetryPolicy((long)this.metadataStoreConfig.getSessionTimeoutMillis(), (long)this.metadataStoreConfig.getSessionTimeoutMillis(), 0)).build();){
                if (chrootZk.exists(chrootPath, false) == null) {
                    ZKMetadataStore.createFullPathOptimistic(chrootZk, chrootPath, new byte[0], CreateMode.PERSISTENT);
                    log.info("Created zookeeper chroot path {} successfully", (Object)chrootPath);
                }
            }
            catch (Exception e) {
                return FutureUtil.failedFuture((Throwable)e);
            }
        }
        return CompletableFuture.completedFuture(null);
    }

    private static void asyncCreateFullPathOptimistic(ZooKeeper zk, String originalPath, byte[] data, CreateMode createMode, AsyncCallback.StringCallback callback) {
        zk.create(originalPath, data, (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, createMode, (rc, path, ctx, name) -> {
            if (rc != KeeperException.Code.NONODE.intValue()) {
                callback.processResult(rc, path, ctx, name);
            } else {
                String parent = new File(originalPath).getParent().replace("\\", "/");
                ZKMetadataStore.asyncCreateFullPathOptimistic(zk, parent, new byte[0], CreateMode.CONTAINER, (rc1, path1, ctx1, name1) -> {
                    if (rc1 != KeeperException.Code.OK.intValue() && rc1 != KeeperException.Code.NODEEXISTS.intValue()) {
                        callback.processResult(rc1, path1, ctx1, name1);
                    } else {
                        ZKMetadataStore.asyncCreateFullPathOptimistic(zk, originalPath, data, createMode, callback);
                    }
                });
            }
        }, null);
    }

    private static void createFullPathOptimistic(ZooKeeper zkc, String path, byte[] data, CreateMode createMode) throws KeeperException, InterruptedException {
        CountDownLatch latch = new CountDownLatch(1);
        AtomicInteger rc = new AtomicInteger(KeeperException.Code.OK.intValue());
        ZKMetadataStore.asyncCreateFullPathOptimistic(zkc, path, data, createMode, (rc2, path1, ctx, name) -> {
            rc.set(rc2);
            latch.countDown();
        });
        latch.await();
        if (rc.get() != KeeperException.Code.OK.intValue()) {
            throw KeeperException.create((KeeperException.Code)KeeperException.Code.get((int)rc.get()));
        }
    }

    public String getRootPath() {
        return this.rootPath;
    }
}

