/*
 * Decompiled with CFR 0.152.
 */
package io.stargate.bridge.grpc;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.time.Duration;
import java.time.Period;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAmount;
import java.time.temporal.TemporalUnit;
import java.time.temporal.UnsupportedTemporalTypeException;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class CqlDuration
implements TemporalAmount {
    static final long NANOS_PER_MICRO = 1000L;
    static final long NANOS_PER_MILLI = 1000000L;
    static final long NANOS_PER_SECOND = 1000000000L;
    static final long NANOS_PER_MINUTE = 60000000000L;
    static final long NANOS_PER_HOUR = 3600000000000L;
    static final int DAYS_PER_WEEK = 7;
    static final int MONTHS_PER_YEAR = 12;
    private static final Pattern STANDARD_PATTERN = Pattern.compile("\\G(\\d+)(y|Y|mo|MO|mO|Mo|w|W|d|D|h|H|s|S|ms|MS|mS|Ms|us|US|uS|Us|\u00b5s|\u00b5S|ns|NS|nS|Ns|m|M)");
    private static final Pattern ISO8601_PATTERN = Pattern.compile("P((\\d+)Y)?((\\d+)M)?((\\d+)D)?(T((\\d+)H)?((\\d+)M)?((\\d+)S)?)?");
    private static final Pattern ISO8601_WEEK_PATTERN = Pattern.compile("P(\\d+)W");
    private static final Pattern ISO8601_ALTERNATIVE_PATTERN = Pattern.compile("P(\\d{4})-(\\d{2})-(\\d{2})T(\\d{2}):(\\d{2}):(\\d{2})");
    private static final ImmutableList<TemporalUnit> TEMPORAL_UNITS = ImmutableList.of(ChronoUnit.MONTHS, ChronoUnit.DAYS, ChronoUnit.NANOS);
    private final int months;
    private final int days;
    private final long nanoseconds;

    private CqlDuration(int months, int days, long nanoseconds) {
        if (!(months >= 0 && days >= 0 && nanoseconds >= 0L || months <= 0 && days <= 0 && nanoseconds <= 0L)) {
            throw new IllegalArgumentException(String.format("All values must be either negative or positive, got %d months, %d days, %d nanoseconds", months, days, nanoseconds));
        }
        this.months = months;
        this.days = days;
        this.nanoseconds = nanoseconds;
    }

    public static CqlDuration newInstance(int months, int days, long nanoseconds) {
        return new CqlDuration(months, days, nanoseconds);
    }

    public static CqlDuration from(String input) {
        String source;
        boolean isNegative = input.startsWith("-");
        String string = source = isNegative ? input.substring(1) : input;
        if (source.startsWith("P")) {
            if (source.endsWith("W")) {
                return CqlDuration.parseIso8601WeekFormat(isNegative, source);
            }
            if (source.contains("-")) {
                return CqlDuration.parseIso8601AlternativeFormat(isNegative, source);
            }
            return CqlDuration.parseIso8601Format(isNegative, source);
        }
        return CqlDuration.parseStandardFormat(isNegative, source);
    }

    private static CqlDuration parseIso8601Format(boolean isNegative, String source) {
        Matcher matcher = ISO8601_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Unable to convert '%s' to a duration", source));
        }
        Builder builder = new Builder(isNegative);
        if (matcher.group(1) != null) {
            builder.addYears(CqlDuration.groupAsLong(matcher, 2));
        }
        if (matcher.group(3) != null) {
            builder.addMonths(CqlDuration.groupAsLong(matcher, 4));
        }
        if (matcher.group(5) != null) {
            builder.addDays(CqlDuration.groupAsLong(matcher, 6));
        }
        if (matcher.group(7) != null) {
            if (matcher.group(8) != null) {
                builder.addHours(CqlDuration.groupAsLong(matcher, 9));
            }
            if (matcher.group(10) != null) {
                builder.addMinutes(CqlDuration.groupAsLong(matcher, 11));
            }
            if (matcher.group(12) != null) {
                builder.addSeconds(CqlDuration.groupAsLong(matcher, 13));
            }
        }
        return builder.build();
    }

    private static CqlDuration parseIso8601AlternativeFormat(boolean isNegative, String source) {
        Matcher matcher = ISO8601_ALTERNATIVE_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Unable to convert '%s' to a duration", source));
        }
        return new Builder(isNegative).addYears(CqlDuration.groupAsLong(matcher, 1)).addMonths(CqlDuration.groupAsLong(matcher, 2)).addDays(CqlDuration.groupAsLong(matcher, 3)).addHours(CqlDuration.groupAsLong(matcher, 4)).addMinutes(CqlDuration.groupAsLong(matcher, 5)).addSeconds(CqlDuration.groupAsLong(matcher, 6)).build();
    }

    private static CqlDuration parseIso8601WeekFormat(boolean isNegative, String source) {
        Matcher matcher = ISO8601_WEEK_PATTERN.matcher(source);
        if (!matcher.matches()) {
            throw new IllegalArgumentException(String.format("Unable to convert '%s' to a duration", source));
        }
        return new Builder(isNegative).addWeeks(CqlDuration.groupAsLong(matcher, 1)).build();
    }

    private static CqlDuration parseStandardFormat(boolean isNegative, String source) {
        boolean done;
        Matcher matcher = STANDARD_PATTERN.matcher(source);
        if (!matcher.find()) {
            throw new IllegalArgumentException(String.format("Unable to convert '%s' to a duration", source));
        }
        Builder builder = new Builder(isNegative);
        do {
            long number = CqlDuration.groupAsLong(matcher, 1);
            String symbol = matcher.group(2);
            CqlDuration.add(builder, number, symbol);
            boolean bl = done = matcher.end() == source.length();
        } while (matcher.find());
        if (!done) {
            throw new IllegalArgumentException(String.format("Unable to convert '%s' to a duration", source));
        }
        return builder.build();
    }

    private static long groupAsLong(Matcher matcher, int group) {
        return Long.parseLong(matcher.group(group));
    }

    private static Builder add(Builder builder, long number, String symbol) {
        String s2 = symbol.toLowerCase(Locale.ROOT);
        if (s2.equals("y")) {
            return builder.addYears(number);
        }
        if (s2.equals("mo")) {
            return builder.addMonths(number);
        }
        if (s2.equals("w")) {
            return builder.addWeeks(number);
        }
        if (s2.equals("d")) {
            return builder.addDays(number);
        }
        if (s2.equals("h")) {
            return builder.addHours(number);
        }
        if (s2.equals("m")) {
            return builder.addMinutes(number);
        }
        if (s2.equals("s")) {
            return builder.addSeconds(number);
        }
        if (s2.equals("ms")) {
            return builder.addMillis(number);
        }
        if (s2.equals("us") || s2.equals("\u00b5s")) {
            return builder.addMicros(number);
        }
        if (s2.equals("ns")) {
            return builder.addNanos(number);
        }
        throw new IllegalArgumentException(String.format("Unknown duration symbol '%s'", symbol));
    }

    private static long append(StringBuilder builder, long dividend, long divisor, String unit) {
        if (dividend == 0L || dividend < divisor) {
            return dividend;
        }
        builder.append(dividend / divisor).append(unit);
        return dividend % divisor;
    }

    public int getMonths() {
        return this.months;
    }

    public int getDays() {
        return this.days;
    }

    public long getNanoseconds() {
        return this.nanoseconds;
    }

    @Override
    public Temporal addTo(Temporal temporal) {
        return temporal.plus(Period.of(0, this.months, this.days)).plus(Duration.ofNanos(this.nanoseconds));
    }

    @Override
    public Temporal subtractFrom(Temporal temporal) {
        return temporal.minus(Period.of(0, this.months, this.days)).minus(Duration.ofNanos(this.nanoseconds));
    }

    @Override
    public long get(TemporalUnit unit) {
        if (unit == ChronoUnit.MONTHS) {
            return this.months;
        }
        if (unit == ChronoUnit.DAYS) {
            return this.days;
        }
        if (unit == ChronoUnit.NANOS) {
            return this.nanoseconds;
        }
        throw new UnsupportedTemporalTypeException("Unsupported unit: " + unit);
    }

    @Override
    public List<TemporalUnit> getUnits() {
        return TEMPORAL_UNITS;
    }

    public boolean equals(Object other) {
        if (other == this) {
            return true;
        }
        if (other instanceof CqlDuration) {
            CqlDuration that = (CqlDuration)other;
            return this.days == that.days && this.months == that.months && this.nanoseconds == that.nanoseconds;
        }
        return false;
    }

    public int hashCode() {
        return Objects.hash(this.days, this.months, this.nanoseconds);
    }

    public String toString() {
        StringBuilder builder = new StringBuilder();
        if (this.months < 0 || this.days < 0 || this.nanoseconds < 0L) {
            builder.append('-');
        }
        long remainder = CqlDuration.append(builder, Math.abs(this.months), 12L, "y");
        CqlDuration.append(builder, remainder, 1L, "mo");
        CqlDuration.append(builder, Math.abs(this.days), 1L, "d");
        if (this.nanoseconds != 0L) {
            remainder = CqlDuration.append(builder, Math.abs(this.nanoseconds), 3600000000000L, "h");
            remainder = CqlDuration.append(builder, remainder, 60000000000L, "m");
            remainder = CqlDuration.append(builder, remainder, 1000000000L, "s");
            remainder = CqlDuration.append(builder, remainder, 1000000L, "ms");
            remainder = CqlDuration.append(builder, remainder, 1000L, "us");
            CqlDuration.append(builder, remainder, 1L, "ns");
        }
        return builder.toString();
    }

    private static class Builder {
        private final boolean isNegative;
        private int months;
        private int days;
        private long nanoseconds;
        private int currentUnitIndex;

        public Builder(boolean isNegative) {
            this.isNegative = isNegative;
        }

        public Builder addYears(long numberOfYears) {
            this.validateOrder(1);
            this.validateMonths(numberOfYears, 12);
            this.months += (int)numberOfYears * 12;
            return this;
        }

        public Builder addMonths(long numberOfMonths) {
            this.validateOrder(2);
            this.validateMonths(numberOfMonths, 1);
            this.months += (int)numberOfMonths;
            return this;
        }

        public Builder addWeeks(long numberOfWeeks) {
            this.validateOrder(3);
            this.validateDays(numberOfWeeks, 7);
            this.days += (int)numberOfWeeks * 7;
            return this;
        }

        public Builder addDays(long numberOfDays) {
            this.validateOrder(4);
            this.validateDays(numberOfDays, 1);
            this.days += (int)numberOfDays;
            return this;
        }

        public Builder addHours(long numberOfHours) {
            this.validateOrder(5);
            this.validateNanos(numberOfHours, 3600000000000L);
            this.nanoseconds += numberOfHours * 3600000000000L;
            return this;
        }

        public Builder addMinutes(long numberOfMinutes) {
            this.validateOrder(6);
            this.validateNanos(numberOfMinutes, 60000000000L);
            this.nanoseconds += numberOfMinutes * 60000000000L;
            return this;
        }

        public Builder addSeconds(long numberOfSeconds) {
            this.validateOrder(7);
            this.validateNanos(numberOfSeconds, 1000000000L);
            this.nanoseconds += numberOfSeconds * 1000000000L;
            return this;
        }

        public Builder addMillis(long numberOfMillis) {
            this.validateOrder(8);
            this.validateNanos(numberOfMillis, 1000000L);
            this.nanoseconds += numberOfMillis * 1000000L;
            return this;
        }

        public Builder addMicros(long numberOfMicros) {
            this.validateOrder(9);
            this.validateNanos(numberOfMicros, 1000L);
            this.nanoseconds += numberOfMicros * 1000L;
            return this;
        }

        public Builder addNanos(long numberOfNanos) {
            this.validateOrder(10);
            this.validateNanos(numberOfNanos, 1L);
            this.nanoseconds += numberOfNanos;
            return this;
        }

        private void validateMonths(long units, int monthsPerUnit) {
            this.validate(units, (Integer.MAX_VALUE - this.months) / monthsPerUnit, "months");
        }

        private void validateDays(long units, int daysPerUnit) {
            this.validate(units, (Integer.MAX_VALUE - this.days) / daysPerUnit, "days");
        }

        private void validateNanos(long units, long nanosPerUnit) {
            this.validate(units, (Long.MAX_VALUE - this.nanoseconds) / nanosPerUnit, "nanoseconds");
        }

        private void validate(long units, long limit, String unitName) {
            Preconditions.checkArgument(units <= limit, "Invalid duration. The total number of %s must be less or equal to %s", (Object)unitName, Integer.MAX_VALUE);
        }

        private void validateOrder(int unitIndex) {
            if (unitIndex == this.currentUnitIndex) {
                throw new IllegalArgumentException(String.format("Invalid duration. The %s are specified multiple times", this.getUnitName(unitIndex)));
            }
            if (unitIndex <= this.currentUnitIndex) {
                throw new IllegalArgumentException(String.format("Invalid duration. The %s should be after %s", this.getUnitName(this.currentUnitIndex), this.getUnitName(unitIndex)));
            }
            this.currentUnitIndex = unitIndex;
        }

        private String getUnitName(int unitIndex) {
            switch (unitIndex) {
                case 1: {
                    return "years";
                }
                case 2: {
                    return "months";
                }
                case 3: {
                    return "weeks";
                }
                case 4: {
                    return "days";
                }
                case 5: {
                    return "hours";
                }
                case 6: {
                    return "minutes";
                }
                case 7: {
                    return "seconds";
                }
                case 8: {
                    return "milliseconds";
                }
                case 9: {
                    return "microseconds";
                }
                case 10: {
                    return "nanoseconds";
                }
            }
            throw new AssertionError((Object)("unknown unit index: " + unitIndex));
        }

        public CqlDuration build() {
            return this.isNegative ? new CqlDuration(-this.months, -this.days, -this.nanoseconds) : new CqlDuration(this.months, this.days, this.nanoseconds);
        }
    }
}

