package io.squashql.query.database;

import com.clickhouse.client.*;
import io.squashql.ClickHouseDatastore;
import io.squashql.ClickHouseUtil;
import io.squashql.query.ColumnarTable;
import io.squashql.query.Header;
import io.squashql.query.RowTable;
import io.squashql.query.Table;
import org.eclipse.collections.api.tuple.Pair;

import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ExecutionException;

public class ClickHouseQueryEngine extends AQueryEngine<ClickHouseDatastore> {

  /**
   * <a href="https://clickhouse.com/docs/en/sql-reference/aggregate-functions/reference/">aggregate functions</a>
   * NOTE: there is more but only a subset is proposed here.
   */
  public static final List<String> SUPPORTED_AGGREGATION_FUNCTIONS = List.of(
          "count",
          "min",
          "max",
          "sum",
          "avg",
          "any",
          "stddevPop",
          "stddevSamp",
          "varPop",
          "varSamp",
          "covarPop",
          "covarSamp");

  protected final ClickHouseNode node;

  public ClickHouseQueryEngine(ClickHouseDatastore datastore) {
    super(datastore, new ClickHouseQueryRewriter());
    this.node = ClickHouseNode.builder()
            .host(this.datastore.dataSource.getHost())
            .port(ClickHouseProtocol.HTTP, this.datastore.dataSource.getPort())
            .build();
  }

  @Override
  protected Table retrieveAggregates(DatabaseQuery query, String sql) {
    try (ClickHouseClient client = ClickHouseClient.newInstance(ClickHouseProtocol.HTTP);
         ClickHouseResponse response = client.connect(this.node)
                 .format(ClickHouseFormat.RowBinaryWithNamesAndTypes)
                 .query(sql)
                 .execute()
                 .get()) {
      Pair<List<Header>, List<List<Object>>> result = transformToColumnFormat(
              query,
              response.getColumns(),
              (column, name) -> name,
              (column, name) -> ClickHouseUtil.clickHouseTypeToClass(column.getDataType()),
              response.records().iterator(),
              (i, r) -> r.getValue(i).asObject(),
              this.queryRewriter);
      return new ColumnarTable(
              result.getOne(),
              new HashSet<>(query.measures),
              result.getTwo());
    } catch (ExecutionException | InterruptedException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public Table executeRawSql(String sql) {
    try (ClickHouseClient client = ClickHouseClient.newInstance(ClickHouseProtocol.HTTP);
         ClickHouseResponse response = client.connect(this.node)
                 .format(ClickHouseFormat.RowBinaryWithNamesAndTypes)
                 .query(sql)
                 .execute()
                 .get()) {
      Pair<List<Header>, List<List<Object>>> result = transformToRowFormat(
              response.getColumns(),
              column -> column.getColumnName(),
              column -> ClickHouseUtil.clickHouseTypeToClass(column.getDataType()),
              response.records().iterator(),
              (i, r) -> r.getValue(i).asObject());
      return new RowTable(result.getOne(), result.getTwo());
    } catch (ExecutionException | InterruptedException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public List<String> supportedAggregationFunctions() {
    return SUPPORTED_AGGREGATION_FUNCTIONS;
  }

  static class ClickHouseQueryRewriter implements QueryRewriter {

    @Override
    public String fieldName(String field) {
      return SqlUtils.backtickEscape(field);
    }

    @Override
    public String measureAlias(String alias) {
      return SqlUtils.backtickEscape(alias);
    }

    @Override
    public boolean usePartialRollupSyntax() {
      // Not supported as of now: https://github.com/ClickHouse/ClickHouse/issues/322#issuecomment-615087004
      // Tested with version https://github.com/ClickHouse/ClickHouse/tree/v22.10.2.11-stable
      return false;
    }

    @Override
    public boolean useGroupingFunction() {
      return true;
    }
  }
}
