/*
 * Copyright (c) 2016 Erik Håkansson, http://squark.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Copyright (c) 2016 Erik Håkansson, http://squark.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.squark.yggdrasil.maven.plugin;

import com.google.common.base.Joiner;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import io.squark.yggdrasil.core.api.Constants;
import io.squark.yggdrasil.core.api.model.YggdrasilConfiguration;
import io.squark.yggdrasil.core.api.model.YggdrasilDependency;
import io.squark.yggdrasil.core.api.model.ProviderConfiguration;
import io.squark.yggdrasil.core.api.util.Scopes;
import io.squark.yggdrasil.maven.provider.api.YggdrasilDependencyMavenUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.descriptor.PluginDescriptor;
import org.apache.maven.plugins.annotations.*;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.MavenProjectHelper;
import org.codehaus.plexus.util.IOUtil;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.artifact.DefaultArtifactType;
import org.eclipse.aether.collection.CollectRequest;
import org.eclipse.aether.collection.CollectResult;
import org.eclipse.aether.collection.DependencySelector;
import org.eclipse.aether.graph.Dependency;
import org.eclipse.aether.graph.DependencyNode;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.DependencyRequest;
import org.eclipse.aether.util.filter.ScopeDependencyFilter;
import org.eclipse.aether.util.graph.selector.AndDependencySelector;
import org.eclipse.aether.util.graph.selector.ExclusionDependencySelector;
import org.eclipse.aether.util.graph.selector.OptionalDependencySelector;
import org.eclipse.aether.util.graph.selector.ScopeDependencySelector;
import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.nio.file.attribute.FileTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.jar.*;
import java.util.stream.Collectors;

@Mojo(name = "package-yggdrasil", defaultPhase = LifecyclePhase.PACKAGE, requiresProject = true, requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class PackageYggdrasilMojo extends AbstractMojo {

    @Parameter(defaultValue = "${plugin}", readonly = true)
    private PluginDescriptor pluginDescriptor;
    @Parameter(property = "yggdrasil.classesDir", defaultValue = "classes")
    private String classesDir;
    @Parameter(property = "yggdrasil.configFile", defaultValue = "classes/META-INF/yggdrasil.json")
    private String configFile;
    @Parameter(property = "yggdrasil.dependencyResolutionProvider", defaultValue = "maven")
    private String dependencyResolutionProviderString;
    /**
     * The MavenProject object.
     *
     * @parameter expression="${project}"
     * @readonly
     */
    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    private MavenProject project;
    @Component
    private MavenProjectHelper mavenProjectHelper;
    /**
     * The entry point to Aether, i.e. the component doing all the work.
     *
     * @component
     */
    @Parameter(defaultValue = "${repositorySystem}")
    @Component
    private RepositorySystem repositorySystem;
    /**
     * The current repository/network configuration of Maven.
     *
     * @parameter default-value="${repositorySystemSession}"
     * @readonly
     */
    @Parameter(readonly = true, defaultValue = "${repositorySystemSession}")
    private RepositorySystemSession repositorySystemSession;
    /**
     * The project's remote repositories to use for the resolution of plugins and their
     * dependencies.
     *
     * @parameter default-value="${project.remotePluginRepositories}"
     * @readonly
     */
    @Parameter(defaultValue = "${project.remotePluginRepositories}", readonly = true)
    private List<RemoteRepository> remoteRepositories;
    /**
     * Location of the file.
     */
    @Parameter(defaultValue = "${project.build.directory}", property = "outputDir", required = true)
    private File outputDirectory;
    @Parameter(property = "yggdrasil.manifest")
    private Map<String, Object> manifest;

    @Parameter(property = "yggdrasil.mainClass")
    private String mainClass;

    @Parameter(property = "yggdrasil.exclusions")
    private List<String> exclusions;

    @Parameter(property = "yggdrasil.includeLogger", defaultValue = "true")
    private boolean includeLogger;

    @Parameter(property = "yggdrasil.providerConfigurations")
    private HashSet<ProviderConfiguration> providerConfigurations;

    @Parameter(property = "yggdrasil.loadTransitiveProvidedDependencies", defaultValue = "false")
    private boolean loadTransitiveProvidedDependencies;

    private List<String> addedResources = new ArrayList<>();

    private Multimap<String, String> duplicates = HashMultimap.create();
    private Map<String, AddedTarget> addedJars = new HashMap<>();

    @Parameter(property = "yggdrasil.nestedJarClassLoaderVersion", defaultValue = Constants.NESTED_JAR_CLASSLOADER_DEFAULT_VERSION)
    private String nestedJarClassLoaderVersion;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {
        File configFile = buildConfigurationFile();
        Manifest manifest = generateAndReturnManifest();
        try (JarOutputStream targetJarOutputStream = createTargetJar(manifest)) {
            addProjectArtifactAndConfig(targetJarOutputStream, configFile);
            addCompileDependencies(targetJarOutputStream);
            addSelfDependencies(targetJarOutputStream);
            targetJarOutputStream.close();
            logDuplicates();
            mavenProjectHelper.attachArtifact(project, project.getArtifact().getType(), "yggdrasil",
                    new File(outputDirectory + "/" + getTargetJarName()));
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to create target jar", e);
        }
    }

    private void addProjectArtifactAndConfig(JarOutputStream targetJarOutputStream, File configFile) throws
            MojoExecutionException
    {
        File baseDir = new File(outputDirectory, classesDir);
        if (baseDir.exists()) {
            try {
                File artifactFile;
                if ((artifactFile = project.getArtifact().getFile()) == null) {
                    throw new MojoExecutionException("Could not find project artifact. Ran goal before package phase?");
                }
                String name = project.getArtifactId() + "-" + project.getVersion();
                JarEntry localJarEntry = new JarEntry(Constants.LIB_PATH + name + "-classes.jar");
                localJarEntry.setLastModifiedTime(FileTime.fromMillis(artifactFile.lastModified()));
                addResource(new FileInputStream(artifactFile), localJarEntry, targetJarOutputStream);
                JarEntry configFileEntry = new JarEntry("META-INF/" + configFile.getName());
                configFileEntry.setLastModifiedTime(FileTime.fromMillis(artifactFile.lastModified()));
                addResource(new FileInputStream(configFile), configFileEntry, targetJarOutputStream);
            } catch (IOException e) {
                throw new MojoExecutionException("Failed to build local dependency jar", e);
            }
        }
    }

    private String getTargetJarName() {
        return project.getArtifactId() + "-" + project.getVersion() + "-yggdrasil." + project.getPackaging();
    }

    private JarOutputStream createTargetJar(Manifest manifest) throws IOException {
        String jarName = getTargetJarName();
        return new JarOutputStream(new FileOutputStream(outputDirectory + "/" + jarName), manifest);
    }

    /**
     * modified from http://stackoverflow.com/a/1281295/961395
     */
    private void add(JarFile sourceJarFile, JarEntry sourceJarEntry, JarOutputStream targetJarOutputStream,
            String sourceName) throws MojoExecutionException
    {
        if (sourceJarEntry.isDirectory()) {
            addDirectory(sourceJarEntry.getName(), targetJarOutputStream);
        } else {
            addResource(sourceJarFile, sourceJarEntry, targetJarOutputStream, sourceName);
        }
    }

    /*
    Partly stolen from
    https://github.com/apache/maven-plugins/blob/trunk/maven-shade-plugin/src/main/java/org
    /apache/maven/plugins/shade/DefaultShader.java @ 2016-03-24
     */
    private void logDuplicates() {

        Multimap<Collection<String>, String> overlapping = HashMultimap.create(20, 15);

        for (String file : duplicates.keySet()) {
            Collection<String> resources = duplicates.get(file);
            if (resources.size() > 1) {
                overlapping.put(resources, file);
            }
        }

        getLog().warn("Some resources are contained in two or more JARs. This is usually safe put may cause" +
                " undefined behaviour if different versions of resources are expected");
        for (Collection<String> jarz : overlapping.keySet()) {
            List<String> jarzStrings = new LinkedList<>();

            for (String file : jarz) {
                jarzStrings.add(file);
            }

            List<String> classes = overlapping.get(jarz).stream().map(clazz -> clazz.replace(".class", "")).collect(
                    Collectors.toCollection(LinkedList::new));

            getLog().warn(Joiner.on(", ").join(jarzStrings) + " define " + classes.size() + " overlapping resource(s): ");
            int max = 10;
            for (int i = 0; i < Math.min(max, classes.size()); i++) {
                getLog().warn("  - " + classes.get(i));
            }

            if (classes.size() > max) {
                getLog().warn("  - " + (classes.size() - max) + " more...");
            }

        }
    }

    private void addResource(File file, JarOutputStream targetJarOutputStream, String targetName) throws MojoExecutionException {
        if (addedResources.contains(file.getName())) {
            return;
        }
        addedResources.add(file.getName());
        JarEntry targetJarEntry = new JarEntry(targetName);
        targetJarEntry.setTime(file.lastModified());
        try (FileInputStream inputStream = new FileInputStream(file)) {
            addResource(inputStream, targetJarEntry, targetJarOutputStream);
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to add jar " + file.getPath(), e);
        }
    }

    private void addResource(JarFile sourceJarFile, JarEntry sourceJarEntry, JarOutputStream targetJarOutputStream,
            String sourceName) throws MojoExecutionException
    {
        String name = sourceJarEntry.getName();
        duplicates.put(name, sourceName);
        if (addedResources.contains(name)) {
            return;
        }
        if (name.matches("META-INF\\/.*\\.(RSA|SF|DSA)")) {
            getLog().warn("Excluding " + name + " to workaround signature issues.");
            return;
        }
        JarEntry targetJarEntry = new JarEntry(name);
        targetJarEntry.setTime(sourceJarEntry.getTime());
        try (InputStream in = sourceJarFile.getInputStream(sourceJarEntry)) {
            addResource(in, targetJarEntry, targetJarOutputStream);
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to copy jar entry " + name + " from jar " + sourceJarFile.getName(), e);
        }
        addedResources.add(name);
    }

    private void addResource(InputStream in, JarEntry targetJarEntry, JarOutputStream targetJarOutputStream) throws IOException {
        targetJarOutputStream.putNextEntry(targetJarEntry);
        IOUtil.copy(in, targetJarOutputStream);
        targetJarOutputStream.closeEntry();
    }

    private void addDirectory(String name, JarOutputStream targetJarOutputStream) throws MojoExecutionException {
        if (name.lastIndexOf('/') > 0) {
            String parent = name.substring(0, name.lastIndexOf('/'));

            if (!addedResources.contains(parent)) {
                addDirectory(parent, targetJarOutputStream);
            }
        }
        if (!name.endsWith("/")) {
            name += "/";
        }
        if (addedResources.contains(name)) {
            return;
        }

        JarEntry entry = new JarEntry(name);
        try {
            targetJarOutputStream.putNextEntry(entry);
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to add directory " + name, e);
        }
        addedResources.add(name);
    }

    private Manifest generateAndReturnManifest() throws MojoExecutionException {
        File manifestFile = new File(new File(outputDirectory, classesDir), "META-INF/MANIFEST.MF");
        Manifest manifest;
        if (manifestFile.exists()) {
            try {
                FileInputStream fileInputStream = new FileInputStream(manifestFile);
                manifest = new Manifest(fileInputStream);
                fileInputStream.close();
            } catch (IOException e) {
                getLog().error(e);
                throw new MojoExecutionException(e.getMessage());
            }
        } else {
            manifest = new Manifest();
            manifest.getMainAttributes().put(Attributes.Name.MANIFEST_VERSION, "1.0");

            manifest.getMainAttributes().put(new Attributes.Name("Build-Jdk"), System.getProperty("java.version"));
        }
        manifest.getMainAttributes().put(Attributes.Name.MAIN_CLASS, Constants.YGGDRASIL_BOOTSTRAP_CLASS_NAME);
        manifest.getMainAttributes().put(new Attributes.Name("Yggdrasil-Version"), pluginDescriptor.getVersion());
        return manifest;
    }

    private File buildConfigurationFile() throws MojoExecutionException {

        YggdrasilConfiguration yggdrasilConfiguration = new YggdrasilConfiguration();
        YggdrasilDependency yggdrasilDependency = getProjectProvidedDependencies();
        yggdrasilConfiguration.setDependencies(yggdrasilDependency.getChildDependencies());
        yggdrasilConfiguration.setYggdrasilVersion(pluginDescriptor.getVersion());
        yggdrasilConfiguration.setLoadTransitiveProvidedDependencies(loadTransitiveProvidedDependencies);

        String classesJar = Constants.LIB_PATH + project.getArtifactId() + "-" + project.getVersion() + "-classes.jar";
        yggdrasilConfiguration.setClassesJar(classesJar);

        if (StringUtils.isNotEmpty(mainClass)) {
            yggdrasilConfiguration.setMainClass(mainClass);
        }

        validateProviderConfigurations();
        yggdrasilConfiguration.setProviderConfigurations(providerConfigurations);

        Gson gson = new GsonBuilder().setPrettyPrinting().create();
        String json = gson.toJson(yggdrasilConfiguration);

        File configurationFile = new File(outputDirectory, configFile);
        if (!configurationFile.exists()) {
            try {
                configurationFile.getParentFile().mkdirs();
                configurationFile.createNewFile();
            } catch (IOException e) {
                getLog().error(e);
                throw new MojoExecutionException(e.getMessage());
            }
        }
        try {
            FileOutputStream outputStream = new FileOutputStream(configurationFile);
            outputStream.write(json.getBytes());
            outputStream.close();
        } catch (IOException e) {
            getLog().error(e);
            throw new MojoExecutionException(e.getMessage());
        }
        return configurationFile;
    }

    private void validateProviderConfigurations() throws MojoExecutionException {
        if (providerConfigurations != null) {
            for (ProviderConfiguration providerConfiguration : providerConfigurations) {
                if (providerConfiguration.getIdentifier() == null) {
                    throw new MojoExecutionException("ProviderConfiguration must supply identifier");
                }
                if (providerConfiguration.getProperties() == null) {
                    throw new MojoExecutionException("ProviderConfiguration must supply properties");
                }
            }
        }
    }

    private void addCompileDependencies(JarOutputStream jarOutputStream) throws MojoExecutionException, MojoFailureException,
            IOException
    {

        List<org.apache.maven.model.Dependency> dependencies = project.getDependencies();

        for (org.apache.maven.model.Dependency dependency : dependencies) {
            addDependency(dependency, jarOutputStream, true, false, Constants.LIB_PATH, null);
        }
    }

    private DependencyNode resolveDependency(org.apache.maven.model.Dependency dependency) throws MojoExecutionException {
        CollectRequest collectRequest = new CollectRequest();
        collectRequest.setRoot(new Dependency(
                new DefaultArtifact(dependency.getGroupId(), dependency.getArtifactId(), dependency.getClassifier(), null,
                        dependency.getVersion(), new DefaultArtifactType(dependency.getType())), dependency.getScope()));
        collectRequest.setRepositories(remoteRepositories);
        try {
            CollectResult collectResult = repositorySystem.collectDependencies(repositorySystemSession, collectRequest);
            DependencyNode node = collectResult.getRoot();

            DependencyRequest dependencyRequest = new DependencyRequest();
            dependencyRequest.setFilter(new ScopeDependencyFilter(null));
            dependencyRequest.setRoot(node);
            repositorySystem.resolveDependencies(repositorySystemSession, dependencyRequest);
            return node;
        } catch (Exception e) {
            getLog().error(e);
            throw new MojoExecutionException(e.getMessage());
        }
    }

    private DependencyNode addDependency(org.apache.maven.model.Dependency dependency, JarOutputStream jarOutputStream,
            boolean asJar, boolean addAsRef, String pathIfJar, @Nullable String jarOverrideName) throws MojoExecutionException
    {
        try {
            DependencyNode node = resolveDependency(dependency);

            if (asJar) {
                addNodeAsJar(node, jarOutputStream, addAsRef, pathIfJar, jarOverrideName);
            } else {
                addNodeAsClasses(node, jarOutputStream);
            }
            return node;
        } catch (Exception e) {
            getLog().error(e);
            throw new MojoExecutionException(e.getMessage());
        }
    }

    private void addSelfDependencies(JarOutputStream targetJarOutputStream) throws MojoFailureException, MojoExecutionException {
        addDependency(getApiDependency(), targetJarOutputStream, true, false, Constants.LIB_PATH, null);
        addDependency(getCoreDependency(), targetJarOutputStream, true, true, Constants.YGGDRASIL_RUNTIME_LIB_PATH, null);
        addDependency(getDependencyResolutionProvider(), targetJarOutputStream, true, true, Constants.YGGDRASIL_RUNTIME_LIB_PATH,
                null);
        String fallbackName = Constants.YGGDRASIL_LOGGING_FALLBACK_ARTIFACT_ID + ".jar";
        addDependency(getLoggingFallback(), targetJarOutputStream, true, false, Constants.YGGDRASIL_RUNTIME_OPTIONAL_LIB_PATH,
                fallbackName);
        addDependency(getBootstrapDependency(), targetJarOutputStream, false, false, null, null);
        addDependency(getClassLoaderDependency(), targetJarOutputStream, false, false, null, null);
    }

    private org.apache.maven.model.Dependency getBootstrapDependency() {
        String yggdrasilVersion = pluginDescriptor.getVersion();
        org.apache.maven.model.Dependency coreDependency = new org.apache.maven.model.Dependency();
        coreDependency.setGroupId(Constants.YGGDRASIL_BOOTSTRAP_GROUP_ID);
        coreDependency.setArtifactId(Constants.YGGDRASIL_BOOTSTRAP_ARTIFACT_ID);
        coreDependency.setVersion(yggdrasilVersion);
        coreDependency.setScope(Scopes.COMPILE);
        return coreDependency;
    }

    @SuppressWarnings("Duplicates")
    private org.apache.maven.model.Dependency getClassLoaderDependency() {
        String yggdrasilVersion = pluginDescriptor.getVersion();
        org.apache.maven.model.Dependency coreDependency = new org.apache.maven.model.Dependency();
        coreDependency.setGroupId(Constants.NESTED_JAR_CLASSLOADER_GROUP_ID);
        coreDependency.setArtifactId(Constants.NESTED_JAR_CLASSLOADER_ARTIFACT_ID);
        coreDependency.setVersion(nestedJarClassLoaderVersion);
        coreDependency.setScope(Scopes.COMPILE);
        return coreDependency;
    }

    @SuppressWarnings("Duplicates")
    private org.apache.maven.model.Dependency getCoreDependency() {
        String yggdrasilVersion = pluginDescriptor.getVersion();
        org.apache.maven.model.Dependency coreDependency = new org.apache.maven.model.Dependency();
        coreDependency.setGroupId(Constants.YGGDRASIL_GROUP_ID);
        coreDependency.setArtifactId(Constants.YGGDRASIL_CORE_ARTIFACT_ID);
        coreDependency.setVersion(yggdrasilVersion);
        coreDependency.setScope(Scopes.COMPILE);
        return coreDependency;
    }

    @SuppressWarnings("Duplicates")
    private org.apache.maven.model.Dependency getApiDependency() {
        String yggdrasilVersion = pluginDescriptor.getVersion();
        org.apache.maven.model.Dependency apiDependency = new org.apache.maven.model.Dependency();
        apiDependency.setGroupId(Constants.YGGDRASIL_GROUP_ID);
        apiDependency.setArtifactId(Constants.YGGDRASIL_API_ARTIFACT_ID);
        apiDependency.setVersion(yggdrasilVersion);
        apiDependency.setScope(Scopes.COMPILE);
        return apiDependency;
    }

    private org.apache.maven.model.Dependency getDependencyResolutionProvider() throws MojoFailureException {
        String groupId;
        String artifactId;
        String version;
        switch (dependencyResolutionProviderString) {
            case "maven":
                groupId = Constants.YGGDRASIL_MAVEN_PROVIDER_GROUP_ID;
                artifactId = Constants.YGGDRASIL_MAVEN_PROVIDER_ARTIFACT_ID;
                version = pluginDescriptor.getVersion();
                break;
            default:
                String[] descriptor = dependencyResolutionProviderString.split(":");
                if (descriptor.length != 3) {
                    throw new MojoFailureException(
                            "Bad format dependencyResolutionProvider. Expected " + "\"groupId:artifactId:version\". Got " +
                                    dependencyResolutionProviderString);
                }
                groupId = descriptor[0];
                artifactId = descriptor[1];
                version = descriptor[2];
                break;
        }
        org.apache.maven.model.Dependency dependencyResolutionProviderDependency = new org.apache.maven.model.Dependency();
        dependencyResolutionProviderDependency.setGroupId(groupId);
        dependencyResolutionProviderDependency.setArtifactId(artifactId);
        dependencyResolutionProviderDependency.setVersion(version);
        dependencyResolutionProviderDependency.setScope(Scopes.COMPILE);
        return dependencyResolutionProviderDependency;
    }

    private org.apache.maven.model.Dependency getLoggingFallback() {
        org.apache.maven.model.Dependency loggingFallbackDependency = new org.apache.maven.model.Dependency();
        String yggdrasilVersion = pluginDescriptor.getVersion();
        loggingFallbackDependency.setGroupId(Constants.YGGDRASIL_LOGGING_API_GROUP_ID);
        loggingFallbackDependency.setArtifactId(Constants.YGGDRASIL_LOGGING_FALLBACK_ARTIFACT_ID);
        loggingFallbackDependency.setVersion(yggdrasilVersion);
        loggingFallbackDependency.setScope(Scopes.COMPILE);
        return loggingFallbackDependency;
    }

    private YggdrasilDependency getProjectProvidedDependencies() throws MojoExecutionException {

        List<org.apache.maven.model.Dependency> mavenDependencies = project.getDependencies();
        if (exclusions == null || exclusions.size() == 0) {
            exclusions = new ArrayList<>();
        }

        YggdrasilDependency rootDependency = new YggdrasilDependency(project.getGroupId(), project.getArtifactId(), null,
                project.getVersion(), null);

        Map<Artifact, String> dependencyArtifacts = new ConcurrentHashMap<>();
        for (org.apache.maven.model.Dependency dependency : mavenDependencies) {
            if (Scopes.PROVIDED.equals(dependency.getScope()) && !Boolean.parseBoolean(dependency.getOptional())) {
                dependencyArtifacts.put(
                        new DefaultArtifact(dependency.getGroupId(), dependency.getArtifactId(), dependency.getClassifier(), null,
                                dependency.getVersion(), new DefaultArtifactType(dependency.getType())), dependency.getScope());
            }
        }
        DefaultRepositorySystemSession repositorySystemSession = new DefaultRepositorySystemSession(this.repositorySystemSession);
        DependencySelector depFilter = new AndDependencySelector(
                new ScopeDependencySelector(Scopes.TEST, Scopes.COMPILE, Scopes.RUNTIME, Scopes.SYSTEM),
                new OptionalDependencySelector(), new ExclusionDependencySelector());
        repositorySystemSession.setDependencySelector(depFilter);
        for (Map.Entry<Artifact, String> entry : dependencyArtifacts.entrySet()) {
            CollectRequest collectRequest = new CollectRequest();
            collectRequest.setRoot(new Dependency(entry.getKey(), entry.getValue()));
            collectRequest.setRepositories(remoteRepositories);
            YggdrasilDependency yggdrasilDependency;
            try {
                CollectResult collectResult = repositorySystem.collectDependencies(repositorySystemSession, collectRequest);
                DependencyNode node = collectResult.getRoot();

                yggdrasilDependency = YggdrasilDependencyMavenUtil.fromDependencyNode(node, exclusions);

            } catch (Exception e) {
                getLog().error(e);
                throw new MojoExecutionException(e.getMessage());
            }
            rootDependency.addChildDependency(yggdrasilDependency);
        }

        return rootDependency;
    }

    private void addNodeAsClasses(DependencyNode node, JarOutputStream targetJarOutputStream) throws MojoExecutionException {
        JarFile jar;
        if (!Scopes.COMPILE.equals(node.getDependency().getScope()) || node.getDependency().getOptional()) {
            return;
        }
        try {
            jar = new JarFile(node.getArtifact().getFile());
        } catch (IOException e) {
            throw new MojoExecutionException("Failed to get jar " + node.getArtifact().getFile().getPath(), e);
        }
        Enumeration enumEntries = jar.entries();
        while (enumEntries.hasMoreElements()) {
            JarEntry sourceEntry = (JarEntry) enumEntries.nextElement();
            String name = sourceEntry.getName();
            if (name.startsWith("java/") || name.startsWith("com/sun/") || name.startsWith("javax/") ||
                    "META-INF/INDEX.LIST".equals(name) || "META-INF/MANIFEST.MF".equals(name))
            {
                continue;
            }
            add(jar, sourceEntry, targetJarOutputStream, node.getArtifact().getFile().getName());
        }
        for (DependencyNode child : node.getChildren()) {
            addNodeAsClasses(child, targetJarOutputStream);
        }
    }

    private void addNodeAsJar(DependencyNode node, JarOutputStream targetJarOutputStream, boolean addAsRef, String path,
            String jarOverrideName) throws MojoExecutionException
    {
        if (!Scopes.COMPILE.equals(node.getDependency().getScope()) || node.getDependency().getOptional()) {
            return;
        }
        if (!path.endsWith("/")) {
            path = path + "/";
        }
        getLog().debug("Including dependency jar " + node.getArtifact().getFile().getPath());
        File file = node.getArtifact().getFile();
        String target = path + (jarOverrideName != null ? jarOverrideName : file.getName());
        if (addAsRef && addedJars.containsKey(file.getName())) {
            String refName = target.replace(".jar", ".ref");
            if (addedResources.contains(refName)) {
                return;
            }
            addedResources.add(refName);
            AddedTarget addedTarget = addedJars.get(file.getName());
            String refContent = addedTarget.path + addedTarget.name;
            InputStream refInputStream = new ByteArrayInputStream(refContent.getBytes());
            JarEntry targetJarEntry = new JarEntry(refName);
            try {
                addResource(refInputStream, targetJarEntry, targetJarOutputStream);
            } catch (IOException e) {
                throw new MojoExecutionException("Failed to add ref", e);
            }
        } else {
            addResource(file, targetJarOutputStream, target);
            addedJars.put(file.getName(), new AddedTarget(path, file.getName()));
        }
        for (DependencyNode child : node.getChildren()) {
            addNodeAsJar(child, targetJarOutputStream, addAsRef, path, null);
        }
    }

    private class AddedTarget {
        String path;
        String name;

        AddedTarget(String path, String name) {
            this.path = path;
            this.name = name;
        }
    }
}
