/*
 * Copyright (c) 2010-2015 Pivotal Software, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You
 * may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License. See accompanying
 * LICENSE file.
 */

package hydra;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.gemstone.gemfire.LogWriter;
import com.gemstone.gemfire.cache.AttributesFactory;
import com.gemstone.gemfire.cache.Cache;
import com.gemstone.gemfire.cache.CacheException;
import com.gemstone.gemfire.cache.CacheFactory;
import com.gemstone.gemfire.cache.DynamicRegionFactory;
import com.gemstone.gemfire.cache.util.GatewayHub;
import com.gemstone.gemfire.cache.util.Gateway;
import com.gemstone.gemfire.cache.Region;
import com.gemstone.gemfire.cache.RegionAttributes;
import com.gemstone.gemfire.cache.RegionExistsException;
import com.gemstone.gemfire.cache.client.Pool;
import com.gemstone.gemfire.cache.client.PoolFactory;
import com.gemstone.gemfire.cache.execute.Function;
import com.gemstone.gemfire.cache.server.CacheServer;
import com.gemstone.gemfire.distributed.DistributedSystem;
import com.gemstone.gemfire.internal.cache.PartitionAttributesImpl;
import com.gemstone.gemfire.internal.cache.xmlcache.BridgeServerCreation;
import com.gemstone.gemfire.internal.cache.xmlcache.CacheCreation;
import com.gemstone.gemfire.internal.cache.xmlcache.CacheXmlGenerator;
import com.gemstone.gemfire.internal.cache.xmlcache.CacheXmlParser;
import com.gemstone.gemfire.internal.cache.xmlcache.FunctionServiceCreation;

/**
 * Helps clients use {@link CacheDescription}.  Methods are thread-safe.
 */
public class CacheHelper {

  /** Tag used in pre-existing XML files to represent autogenerated ports. */
  public static String CACHE_SERVER_PORT_TAG = "%CACHE_SERVER_PORT%";

  /** Name of the cache description used to create the current cache */
  protected static String TheCacheConfig;

  /** Maps of cache, region, and bridge descriptions used to generate XML */
  private static Map XmlCacheConfigs = new HashMap();
  private static Map XmlGatewayHubConfigs = new HashMap();
  private static Map XmlGatewayConfigs = new HashMap();
  private static Map XmlRegionConfigs = new HashMap();
  private static Map XmlBridgeConfigs = new HashMap();
  private static Map XmlPoolConfigs = new HashMap();
  private static Map XmlFunctionConfigs = new HashMap();
  private static Map XmlDiskStoreConfigs = new HashMap();
  private static Map XmlGatewaySenderConfigs = new HashMap();
  private static Map XmlGatewayReceiverConfigs = new HashMap();
  private static Map XmlAsyncEventQueueConfigs = new HashMap();

//------------------------------------------------------------------------------
// Cache
//------------------------------------------------------------------------------

  /**
   * Creates a cache for the given cache configuration from {@link
   * CachePrms#names}, configured using the corresponding {@link
   * CacheDescription}.  Returns the existing cache if it was previously
   * created and is still open.
   * <p>
   * The distributed system is automatically connected using {@link
   * DistributedSystemHelper#connect}.
   *
   * @throws HydraRuntimeException if an attempt is made to reconfigure an
   *         existing cache.
   */
  public static synchronized Cache createCache(String cacheConfig) {
    if (cacheConfig == null) {
      throw new IllegalArgumentException("cacheConfig cannot be null");
    }
    Cache cache = getCache();
    if (cache == null) {

      // look up the cache configuration
      CacheDescription cd = getCacheDescription(cacheConfig);

      // connect to the distributed system, if necessary
      DistributedSystem ds = DistributedSystemHelper.getDistributedSystem();
      if (ds == null) {
         ds = DistributedSystemHelper.connect();
      }

      // configure and create the cache
      try {
        cache = CacheVersionHelper.configureAndCreateCache(cacheConfig, cd, ds);
      } catch (CacheException e) {
        String s = "Unable to create cache using: " + ds;
        throw new HydraRuntimeException(s, e);
      }

      // continue configuring the cache
      log("Configuring cache from config: " + cacheConfig);
      cd.configure(cache);
      log("Configured cache: " + cacheToString(cache));

      // save the cache config for future reference
      TheCacheConfig = cacheConfig;

    } else if (TheCacheConfig == null) {
      // block attempt to create cache in multiple ways
      String s = "Cache was already created without CacheHelper using"
               + " an unknown, and possibly different, configuration";
      throw new HydraRuntimeException(s);

    } else if (!TheCacheConfig.equals(cacheConfig)) {
      // block attempt to recreate cache with clashing configuration
      String s = "Cache already exists using cache configuration "
               + TheCacheConfig + ", cannot also use " + cacheConfig;
      throw new HydraRuntimeException(s);

    } // else it was already created with this configuration, which is fine

    return cache;
  }

  /**
   * Creates a cache using the given XML configuration file.  Returns the
   * existing cache if it was previously created and is still open.
   * <p>
   * Replaces any occurrences of {@link #CACHE_SERVER_PORT_TAG} in the file
   * with an autogenerated random port.
   * <p>
   * The distributed system is automatically connected using {@link
   * DistributedSystemHelper#connect(String)}.
   *
   * @throws HydraRuntimeException if an attempt is made to reconfigure an
   *         existing cache.
   */
  public static synchronized Cache createCacheFromXml(String cacheXmlFile) {
    if (cacheXmlFile == null) {
      throw new IllegalArgumentException("cacheXmlFile cannot be null");
    }
    String fn = FileUtil.absoluteFilenameFor(
                EnvHelper.expandEnvVars(cacheXmlFile));
    Cache cache = getCache();
    if (cache == null) {
      try {
        // autogenerate port on pre-existing XML file, if needed
        String text = FileUtil.getText(fn);
        if (text.indexOf(CACHE_SERVER_PORT_TAG) != -1) {
          int port = PortHelper.getRandomPort();
          String txt = FileUtil.getText(fn);
          txt = txt.replaceAll(CACHE_SERVER_PORT_TAG, String.valueOf(port));
          fn = System.getProperty("user.dir") + File.separator
             + (new File(fn)).getName();
          FileUtil.writeToFile(fn, txt);
          log("Switching to cache XML file: " + fn
             + " with autogenerated bridge server port: " + port);

          // generate blackboard endpoints
          FileInputStream fis = new FileInputStream(new File(fn));
          CacheXmlParser parser = CacheXmlParser.parse(fis);
          CacheCreation dummyCache = parser.getCacheCreation();
          List dummyCacheServers = dummyCache.getBridgeServers();
          if (dummyCacheServers != null) {
            if (dummyCacheServers.size() > 1) {
              String s = "Multiple bridge servers in single JVM not supported";
              throw new UnsupportedOperationException();
            }
            for (Iterator i = dummyCacheServers.iterator(); i.hasNext();) {
              BridgeServerCreation dummyCacheServer =
                                   (BridgeServerCreation)i.next();
              String[] groups = dummyCacheServer.getGroups();
              port = dummyCacheServer.getPort();
              BridgeHelper.getEndpoint(port, groups);
            }
          }
        }
      } catch (IOException e) {
        throw new HydraRuntimeException(fn, e);
      }

      // connect to the distributed system
      DistributedSystem ds =
                 DistributedSystemHelper.connectWithXml(fn);

      // create the cache, first autogenerating ports if needed
      log("Creating cache from XML: " + fn);
      try {
        cache = CacheFactory.create(ds);
      } catch (CacheException e) {
        String s = "Unable to create cache using: " + ds;
        throw new HydraRuntimeException(s, e);
      }
      log("Created cache from XML");

      // save the cache config for future reference
      TheCacheConfig = fn;

    } else if (TheCacheConfig == null) {
      // block attempt to create cache in multiple ways
      String s = "Cache was already created without CacheHelper using"
               + " an unknown, and possibly different, configuration";
      throw new HydraRuntimeException(s);

    } else if (!TheCacheConfig.equals(fn)) {
      // block attempt to recreate cache with clashing configuration
      String s = "Cache already exists using cache configuration "
               + TheCacheConfig + ", cannot also use " + fn;
      throw new HydraRuntimeException(s);

    } // else it was already created with this configuration, which is fine

    return cache;
  }

  /**
   * Generates XML for the given cache and region configurations from {@link
   * CachePrms#names} and {@link RegionPrms#names}, using the corresponding
   * {@link CacheDescription} and {@link RegionDescription}.  Writes the result
   * to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig, String regionConfig, String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         null /* dynamicRegionConfig */,
                         regionConfig,
                         null /* regionNames */,
                         null /* bridgeConfig */,
                         null /* poolConfig */,
                         null /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         null /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache and region configurations from {@link
   * CachePrms#names} and {@link RegionPrms#names}, using the corresponding
   * {@link CacheDescription} and {@link RegionDescription}.  Creates a region
   * for each region name.  Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig, String regionConfig, List regionNames,
                String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         null /* dynamicRegionConfig */,
                         regionConfig,
                         regionNames /* regionNames */,
                         null /* bridgeConfig */,
                         null /* poolConfig */,
                         null /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         null /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, and pool configurations
   * from {@link CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms
   * #names}, and {@link PoolPrms#names}, using the corresponding {@link
   * CacheDescription}, {@link RegionDescription}, {@link BridgeDescription},
   * and {@link PoolDescription}.  Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig, String regionConfig, String bridgeConfig,
                String poolConfig, String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         null /* dynamicRegionConfig */,
                         regionConfig,
                         null /* regionNames */,
                         bridgeConfig /* bridgeConfig */,
                         poolConfig /* poolConfig */,
                         null /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         null /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, and pool configurations
   * from {@link CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms
   * #names}, and {@link PoolPrms#names}, using the corresponding {@link
   * CacheDescription}, {@link RegionDescription}, {@link BridgeDescription},
   * and {@link PoolDescription}.  Creates a region for each region name.
   * Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig, String regionConfig, List regionNames,
                String bridgeConfig, String poolConfig, String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         null /* dynamicRegionConfig */,
                         regionConfig,
                         regionNames /* regionNames */,
                         bridgeConfig /* bridgeConfig */,
                         poolConfig /* poolConfig */,
                         null /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         null /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, and pool configurations
   * from {@link DynamicRegionFactory.Config}, {@link CachePrms#names}, {@link
   * RegionPrms#names}, {@link BridgePrms#names}, and {@link PoolPrms#names},
   * using the corresponding {@link CacheDescription}, {@link
   * RegionDescription}, {@link BridgeDescription}, and {@link PoolDescription}.
   * Creates a region for each region name.  Writes the result to the specified
   * file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                String regionConfig, List regionNames,
                String bridgeConfig, String poolConfig, String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         dynamicRegionConfig /* dynamicRegionConfig */,
                         regionConfig,
                         regionNames /* regionNames */,
                         bridgeConfig /* bridgeConfig */,
                         poolConfig /* poolConfig */,
                         null /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         null /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, pool, and disk store
   * configurations from {@link DynamicRegionFactory.Config}, {@link
   * CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms#names},
   * {@link PoolPrms#names}, and {@link DiskStorePrms#names}, using the
   * corresponding {@link CacheDescription}, {@link RegionDescription}, {@link
   * BridgeDescription}, {@link PoolDescription}, and {@link
   * DiskStoreDescription}.  Creates a region for each region name.
   * Registers the list of functions.  Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                String regionConfig, List regionNames,
                String bridgeConfig, String poolConfig, String diskStoreConfig,
                List functions, String cacheXmlFile) {
    generateCacheXmlFile(cacheConfig,
                         dynamicRegionConfig /* dynamicRegionConfig */,
                         regionConfig,
                         regionNames /* regionNames */,
                         bridgeConfig /* bridgeConfig */,
                         poolConfig /* poolConfig */,
                         diskStoreConfig /* diskStoreConfig */,
                         null /* gatewaySenderConfig */,
                         null /* gatewayReceiverConfig */,
                         null /* asyncEventQueueConfig */,
                         functions /* functions */,
                         cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, pool, and disk store
   * configurations from {@link DynamicRegionFactory.Config}, {@link
   * CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms#names},
   * {@link PoolPrms#names}, and {@link DiskStorePrms#names}, using the
   * corresponding {@link CacheDescription}, {@link RegionDescription}, {@link
   * BridgeDescription}, {@link PoolDescription}, {@link
   * DiskStoreDescription}, etc.  Creates a region for each region name.
   * Registers the list of functions.  Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void generateCacheXmlFile(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                String regionConfig, List regionNames,
                String bridgeConfig, String poolConfig, String diskStoreConfig,
                String gatewaySenderConfig, String gatewayReceiverConfig,
                String asyncEventQueueConfig,
                List functions, String cacheXmlFile) {
    if (cacheConfig == null) {
      throw new IllegalArgumentException("cacheConfig cannot be null");
    }
    if (cacheXmlFile == null) {
      throw new IllegalArgumentException("cacheXmlFile cannot be null");
    }
    String fn = FileUtil.absoluteFilenameFor(
                EnvHelper.expandEnvVars(cacheXmlFile));
    if (!FileUtil.exists(fn)) {
      StringBuilder buf = new StringBuilder("Generating XML file: ")
                              .append(fn).append(" from");
      if (cacheConfig != null) {
        buf.append("\ncache config: " + cacheConfig);
      }
      if (dynamicRegionConfig != null) {
        buf.append("\ndynamic region config: " + dynamicRegionConfig);
      }
      if (regionConfig != null) {
        buf.append("\nregion config: " + regionConfig);
      }
      if (regionNames != null) {
        buf.append("\nregion names: " + regionNames);
      }
      if (bridgeConfig != null) {
        buf.append("\nbridge config: " + bridgeConfig);
      }
      if (poolConfig != null) {
        buf.append("\npool config: " + poolConfig);
      }
      if (diskStoreConfig != null) {
        buf.append("\ndisk store config: " + diskStoreConfig);
      }
      if (gatewaySenderConfig != null) {
        buf.append("\ngateway sender config: " + gatewaySenderConfig);
      }
      if (gatewayReceiverConfig != null) {
        buf.append("\ngateway receiver config: " + gatewayReceiverConfig);
      }
      if (asyncEventQueueConfig != null) {
        buf.append("\nasync event queue config: " + asyncEventQueueConfig);
      }
      if (functions != null) {
        buf.append("\nfunctions: " + functions);
      }
      log(buf.toString());

      Cache dummyCache = generateDummyCache(cacheConfig,
                                            dynamicRegionConfig, fn);
      generateDummyFunctions(dummyCache, functions, fn);
      generateDummyPool(dummyCache, poolConfig, fn);
      generateDummyDiskStore(dummyCache, diskStoreConfig, fn);
      generateDummyGatewaySender(dummyCache, gatewaySenderConfig, fn);
      generateDummyGatewayReceiver(dummyCache, gatewayReceiverConfig, fn);
      generateDummyAsyncEventQueue(dummyCache, asyncEventQueueConfig, fn);
      generateDummyBridge(dummyCache, bridgeConfig, fn);
      if (regionNames != null && regionNames.size() > 0) {
        for (Iterator i = regionNames.iterator(); i.hasNext();) {
          String regionName = (String)i.next();
          generateDummyRegion(dummyCache, regionConfig, regionName, fn);
        }
      } else {
        generateDummyRegion(dummyCache, regionConfig, null, fn);
      }

      // generate the xml file
      generateCacheXmlFile(fn, dummyCache);

    } else {
      if (XmlCacheConfigs.get(fn) == null) {
        // block attempt to create file in multiple ways
        String s = "Cache XML file was already created without CacheHelper"
                 + " or from a different VM using"
                 + " an unknown, and possibly different, configuration: " + fn;
        throw new HydraRuntimeException(s);

      } else {
        // block attempt to recreate file with clashing configuration
        String cc = (String)XmlCacheConfigs.get(fn);
        if (!cc.equals(cacheConfig)) {
          String s = "Cache XML file already exists using cache configuration "
                   + cc + ", cannot also use " + cacheConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String rc = (String)XmlRegionConfigs.get(fn);
        if ((rc == null && regionConfig != null)
         || (rc != null && regionConfig == null)
         || (rc != null && regionConfig != null && !rc.equals(regionConfig))) {
          String s = "Cache XML file already exists using region configuration "
                   + rc + ", cannot also use " + regionConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String bc = (String)XmlBridgeConfigs.get(fn);
        if ((bc == null && bridgeConfig != null)
         || (bc != null && bridgeConfig == null)
         || (bc != null && bridgeConfig != null && !bc.equals(bridgeConfig))) {
          String s = "Cache XML file already exists using bridge configuration "
                   + bc + ", cannot also use " + bridgeConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String cpc = (String)XmlPoolConfigs.get(fn);
        if ((cpc == null && poolConfig != null)
         || (cpc != null && poolConfig == null)
         || (cpc != null && poolConfig != null && !cpc.equals(poolConfig))) {
          String s = "Cache XML file already exists using pool configuration "
                   + cpc + ", cannot also use " + poolConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String dsc = (String)XmlDiskStoreConfigs.get(fn);
        if ((dsc == null && diskStoreConfig != null)
         || (dsc != null && diskStoreConfig == null)
         || (dsc != null && diskStoreConfig != null
                         && !dsc.equals(diskStoreConfig))) {
          String s = "Cache XML file already exists using disk store "
                   + "configuration " + dsc + ", cannot also use "
                   + diskStoreConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        {
          String c = (String)XmlGatewaySenderConfigs.get(fn);
          if ((c == null && gatewaySenderConfig != null)
           || (c != null && gatewaySenderConfig == null)
           || (c != null && gatewaySenderConfig != null
                         && !c.equals(gatewaySenderConfig))) {
            String s = "Cache XML file already exists using gateway sender configuration " + c + ", cannot also use " + gatewaySenderConfig + ": " + fn;
            throw new HydraRuntimeException(s);
          }
        }
        {
          String c = (String)XmlGatewayReceiverConfigs.get(fn);
          if ((c == null && gatewayReceiverConfig != null)
           || (c != null && gatewayReceiverConfig == null)
           || (c != null && gatewayReceiverConfig != null
                         && !c.equals(gatewayReceiverConfig))) {
            String s = "Cache XML file already exists using gateway receiver configuration " + c + ", cannot also use " + gatewayReceiverConfig + ": " + fn;
            throw new HydraRuntimeException(s);
          }
        }
        {
          String c = (String)XmlAsyncEventQueueConfigs.get(fn);
          if ((c == null && asyncEventQueueConfig != null)
           || (c != null && asyncEventQueueConfig == null)
           || (c != null && asyncEventQueueConfig != null
                         && !c.equals(asyncEventQueueConfig))) {
            String s = "Cache XML file already exists using async event queue configuration " + c + ", cannot also use " + asyncEventQueueConfig + ": " + fn;
            throw new HydraRuntimeException(s);
          }
        }
        List fc = (List)XmlFunctionConfigs.get(fn);
        List functionNames = classnamesFor(functions);
        if ((fc == null && functionNames != null)
         || (fc != null && functionNames == null)
         || (fc != null && functionNames != null
                        && !(fc.containsAll(functionNames)
                             && functionNames.containsAll(fc)))) {
          String s = "Cache XML file already exists using functions "
                   + fc + ", cannot also use " + functions + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        // else it was already created with this configuration, which is fine
        return;
      }
    }
  }

  /**
   * Generates a CacheCreation object (dummyCache) for the given cache, dynamicRegionConfig
   * and GatewayHubConfig using configurations from {@link GatewayHubPrms#names}, 
   * {@link DynamicRegionFactory.Config}, and {@link * CachePrms#names}.
   * Return the CacheCreation object for subsequent use in generateCacheXmlFile() with 
   * gatewayConfig as well as regionConfig, bridgeConfig, poolConfig, etc.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   */
  public static synchronized Cache startCacheXmlGenerationForGateway(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                String gatewayHubConfig, 
                String cacheXmlFile) {
    if (cacheConfig == null) {
      throw new IllegalArgumentException("cacheConfig cannot be null");
    }
    if (gatewayHubConfig == null) {
      throw new IllegalArgumentException("gatewayHubConfig cannot be null");
    }
    if (cacheXmlFile == null) {
      throw new IllegalArgumentException("cacheXmlFile cannot be null");
    }

    String fn = FileUtil.absoluteFilenameFor(
                EnvHelper.expandEnvVars(cacheXmlFile));
    Cache dummyCache = generateDummyCache(cacheConfig, dynamicRegionConfig, fn);
    generateDummyGatewayHub(dummyCache, gatewayHubConfig,fn);
    return dummyCache;
  }

  /**
   * Generates XML for the given cache (the creation object returned by 
   * generateCacheXmlForGatewayHub) with the given gatewayConfig, region, 
   * bridge, pool, and disk store configurations from {@link GatewayPrms#names},
   * {@link CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms#names},
   * {@link PoolPrms#names}, and {@link DiskStorePrms#names}.
   * Writes the result to the specified file.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   *
   * @throws HydraRuntimeException if an attempt is made to rewrite an existing
   *         cache XML file with different content.
   */
  public static synchronized void finishCacheXmlGenerationForGateway(
                Cache cache,
                String cacheConfig,
                String gatewayHubConfig,
                String gatewayConfig,
                String regionConfig, List regionNames,
                String bridgeConfig, String poolConfig, String diskStoreConfig,
                List functions, String cacheXmlFile) {
    if ( !(cache instanceof CacheCreation) ) {
      throw new IllegalArgumentException("cache must be a CacheCreation object");
    }
    if (cacheXmlFile == null) {
      throw new IllegalArgumentException("cacheXmlFile cannot be null");
    }
    String fn = FileUtil.absoluteFilenameFor(
                EnvHelper.expandEnvVars(cacheXmlFile));
    if (!FileUtil.exists(fn)) {
      log("Generating XML file: " + fn + " from gateway config: "
         + gatewayConfig + " and region config " + regionConfig
         + " and bridge config " + bridgeConfig
         + " and pool config " + poolConfig);

      CacheCreation dummyCache = (CacheCreation)cache;
      generateDummyGateways(dummyCache, gatewayConfig, fn);
      generateDummyFunctions(dummyCache, functions, fn);
      generateDummyPool(dummyCache, poolConfig, fn);
      generateDummyDiskStore(dummyCache, diskStoreConfig, fn);
      generateDummyBridge(dummyCache, bridgeConfig, fn);
      if (regionNames != null && regionNames.size() > 0) {
        for (Iterator i = regionNames.iterator(); i.hasNext();) {
          String regionName = (String)i.next();
          generateDummyRegion(dummyCache, regionConfig, regionName, fn);
        }
      } else {
        generateDummyRegion(dummyCache, regionConfig, null, fn);
      }

      // generate the xml file
      generateCacheXmlFile(fn, dummyCache);

    } else {
      if (XmlCacheConfigs.get(fn) == null) {
        // block attempt to create file in multiple ways
        String s = "Cache XML file was already created without CacheHelper"
                 + " or from a different VM using"
                 + " an unknown, and possibly different, configuration: " + fn;
        throw new HydraRuntimeException(s);

      } else {
        // block attempt to recreate file with clashing configuration
        String cc = (String)XmlCacheConfigs.get(fn);
        if (!cc.equals(cacheConfig)) {
          String s = "Cache XML file already exists using cache configuration "
                   + cc + ", cannot also use " + cacheConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String ghc = (String)XmlGatewayHubConfigs.get(fn);
        if ((ghc == null && gatewayHubConfig != null)  
         || (ghc != null && gatewayHubConfig == null) 
         || (ghc != null && gatewayHubConfig != null && !ghc.equals(gatewayHubConfig))) {
          String s = "Cache XML file already exists using gatewayHub configuration "
                   + ghc + ", cannot also use " + gatewayHubConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String gc = (String)XmlGatewayConfigs.get(fn);
        if ((gc == null && gatewayConfig != null)  
         || (gc != null && gatewayConfig == null) 
         || (gc != null && gatewayConfig != null && !gc.equals(gatewayConfig))) {
          String s = "Cache XML file already exists using gateway configuration "
                   + gc + ", cannot also use " + gatewayConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String rc = (String)XmlRegionConfigs.get(fn);
        if ((rc == null && regionConfig != null)
         || (rc != null && regionConfig == null)
	 || (rc != null && regionConfig != null && !rc.equals(regionConfig))) {
          String s = "Cache XML file already exists using region configuration "
                   + rc + ", cannot also use " + regionConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String bc = (String)XmlBridgeConfigs.get(fn);
        if ((bc == null && bridgeConfig != null)
         || (bc != null && bridgeConfig == null)
	 || (bc != null && bridgeConfig != null && !bc.equals(bridgeConfig))) {
          String s = "Cache XML file already exists using bridge configuration "
                   + bc + ", cannot also use " + bridgeConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String cpc = (String)XmlPoolConfigs.get(fn);
        if ((cpc == null && poolConfig != null)
         || (cpc != null && poolConfig == null)
	 || (cpc != null && poolConfig != null && !cpc.equals(poolConfig))) {
          String s = "Cache XML file already exists using pool configuration "
                   + cpc + ", cannot also use " + poolConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        String dsc = (String)XmlDiskStoreConfigs.get(fn);
        if ((dsc == null && diskStoreConfig != null)
         || (dsc != null && diskStoreConfig == null)
	 || (dsc != null && diskStoreConfig != null
                         && !dsc.equals(diskStoreConfig))) {
          String s = "Cache XML file already exists using disk store "
                   + "configuration " + dsc + ", cannot also use "
                   + diskStoreConfig + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        List fc = (List)XmlFunctionConfigs.get(fn);
        List functionNames = classnamesFor(functions);
        if ((fc == null && functionNames != null)
         || (fc != null && functionNames == null)
	 || (fc != null && functionNames != null
                        && !(fc.containsAll(functionNames)
                             && functionNames.containsAll(fc)))) {
          String s = "Cache XML file already exists using functions "
                   + fc + ", cannot also use " + functions + ": " + fn;
          throw new HydraRuntimeException(s);
        }
        // else it was already created with this configuration, which is fine
        return;
      }
    }
  }

  /**
   * Generates XML for the given cache, region, bridge, and pool configurations
   * from {@link DynamicRegionFactory.Config}, {@link CachePrms#names}, {@link
   * RegionPrms#names}, {@link BridgePrms#names}, and {@link PoolPrms#names},
   * using the corresponding {@link CacheDescription}, {@link
   * RegionDescription}s, {@link BridgeDescription}, and {@link
   * PoolDescription}s.  Creates a region for each region configuration and
   * name.  Creates a pool for each pool configuration.  Writes the result to
   * the specified file.
   * <p>
   * If <code>regionNames</code> are provided, the number of regionNames must
   * match the number of <code>regionConfigs</code>.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   * <p>
   * IMPORTANT: Unlike other XML-generating methods, no attempt is made to
   * prevent overwriting an existing XML file with a different configuration.
   *
   * @throws HydraRuntimeException if the number of region names is different
   *         from the number of region configs (when region names are provided).
   */
  public static synchronized void generateCacheXmlFileNoCheck(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                List regionConfigs, List regionNames,
                String bridgeConfig, List poolConfigs, String cacheXmlFile) {
    generateCacheXmlFileNoCheck(cacheConfig,
                                dynamicRegionConfig,
                                regionConfigs,
                                regionNames,
                                bridgeConfig,
                                poolConfigs,
                                null, /* diskStoreConfigs */
                                null, /* gatewaySenderConfig */
                                null, /* gatewayReceiverConfig */
                                null /* asyncEventQueueConfig */,
                                null, /* functions */
                                cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, pool, and disk store
   * configurations from {@link DynamicRegionFactory.Config}, {@link
   * CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms#names},
   * {@link PoolPrms#names}, and {@link DiskStorePrms#names}, using the
   * corresponding {@link CacheDescription}, {@link RegionDescription}s, {@link
   * BridgeDescription}, {@link PoolDescription}s, and {@link
   * DiskStoreDescription}s.  Creates a region for each region configuration
   * and name.  Creates a pool for each pool configuration.  Creates a disk
   * store for each disk store configuration.  Registers the list of functions.
   * Writes the result to the specified file.
   * <p>
   * If <code>regionNames</code> are provided, the number of regionNames must
   * match the number of <code>regionConfigs</code>.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   * <p>
   * IMPORTANT: Unlike other XML-generating methods, no attempt is made to
   * prevent overwriting an existing XML file with a different configuration.
   *
   * @throws HydraRuntimeException if the number of region names is different
   *         from the number of region configs (when region names are provided).
   */
  public static synchronized void generateCacheXmlFileNoCheck(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                List regionConfigs, List regionNames,
                String bridgeConfig, List poolConfigs, List diskStoreConfigs,
                List functions, String cacheXmlFile) {
    generateCacheXmlFileNoCheck(cacheConfig,
                                dynamicRegionConfig,
                                regionConfigs,
                                regionNames,
                                bridgeConfig,
                                poolConfigs,
                                diskStoreConfigs, /* diskStoreConfigs */
                                null, /* gatewaySenderConfig */
                                null, /* gatewayReceiverConfig */
                                null, /* asyncEventQueueConfig */
                                functions, /* functions */
                                cacheXmlFile);
  }

  /**
   * Generates XML for the given cache, region, bridge, pool, and disk store
   * configurations from {@link DynamicRegionFactory.Config}, {@link
   * CachePrms#names}, {@link RegionPrms#names}, {@link BridgePrms#names},
   * {@link PoolPrms#names}, and {@link DiskStorePrms#names}, using the
   * corresponding {@link CacheDescription}, {@link RegionDescription}s, {@link
   * BridgeDescription}, {@link PoolDescription}s, {@link
   * DiskStoreDescription}s, etc. Creates a region for each region configuration
   * and name.  Creates a pool for each pool configuration.  Creates a disk
   * store for each disk store configuration.  Registers the list of functions.
   * Writes the result to the specified file.
   * <p>
   * If <code>regionNames</code> are provided, the number of regionNames must
   * match the number of <code>regionConfigs</code>.
   * <p>
   * IMPORTANT: This method should only be invoked from the VM that will use
   * the result.  Otherwise, runtime-generated artifacts (e.g., ports, disk
   * directories) might not work as required.
   * <p>
   * IMPORTANT: This method is not synchronized across multiple VMs.  It is up
   * to the user to avoid writing to the same filename from multiple VMs.
   * <p>
   * IMPORTANT: Unlike other XML-generating methods, no attempt is made to
   * prevent overwriting an existing XML file with a different configuration.
   *
   * @throws HydraRuntimeException if the number of region names is different
   *         from the number of region configs (when region names are provided).
   */
  public static synchronized void generateCacheXmlFileNoCheck(
                String cacheConfig,
                DynamicRegionFactory.Config dynamicRegionConfig, 
                List regionConfigs, List regionNames,
                String bridgeConfig, List poolConfigs, List diskStoreConfigs,
                String gatewaySenderConfig, String gatewayReceiverConfig,
                String asyncEventQueueConfig,
                List functions, String cacheXmlFile) {
    if (cacheConfig == null) {
      throw new IllegalArgumentException("cacheConfig cannot be null");
    }
    if (cacheXmlFile == null) {
      throw new IllegalArgumentException("cacheXmlFile cannot be null");
    }
    String fn = FileUtil.absoluteFilenameFor(
                EnvHelper.expandEnvVars(cacheXmlFile));
    log("Generating XML file: " + fn + " from cache config: "
       + cacheConfig + " and region configs " + regionConfigs
       + " and bridge config " + bridgeConfig
       + " and pool configs " + poolConfigs);
    if (FileUtil.exists(fn)) {
      log("Overwriting XML file: " + fn
        + " with possibly different configuration");
    }

    Cache dummyCache = generateDummyCache(cacheConfig,
                                          dynamicRegionConfig, fn);
    if (poolConfigs != null) {
      for (Iterator i = poolConfigs.iterator(); i.hasNext();) {
        String poolConfig = (String)i.next();
        generateDummyPool(dummyCache, poolConfig, fn);
      }
    }
    if (diskStoreConfigs != null) {
      for (Iterator i = diskStoreConfigs.iterator(); i.hasNext();) {
        String diskStoreConfig = (String)i.next();
        generateDummyDiskStore(dummyCache, diskStoreConfig, fn);
      }
    }
    if (gatewaySenderConfig != null) {
      generateDummyGatewaySender(dummyCache, gatewaySenderConfig, fn);
    }
    if (gatewayReceiverConfig != null) {
      generateDummyGatewayReceiver(dummyCache, gatewayReceiverConfig, fn);
    }
    if (asyncEventQueueConfig != null) {
      generateDummyAsyncEventQueue(dummyCache, asyncEventQueueConfig, fn);
    }
    generateDummyFunctions(dummyCache, functions, fn);
    generateDummyBridge(dummyCache, bridgeConfig, fn);
    if (regionNames != null && regionNames.size() > 0) {
      if (regionConfigs == null ||
          regionConfigs.size() != regionNames.size()) {
        String s = "Number of regionConfigs (" + regionConfigs
                 + ") != number of regionNames (" + regionNames + ")";
        throw new HydraRuntimeException(s);
      }
      for (int i = 0; i < regionConfigs.size(); i++) {
        String regionConfig = (String)regionConfigs.get(i);
        String regionName = (String)regionNames.get(i);
        generateDummyRegion(dummyCache, regionConfig, regionName, fn);
      }
    } else {
      for (Iterator i = regionConfigs.iterator(); i.hasNext();) {
        String regionConfig = (String)i.next();
        generateDummyRegion(dummyCache, regionConfig, null, fn);
      }
    }

    // generate the xml file
    generateCacheXmlFile(fn, dummyCache);
  }

  /**
   * Generates a dummy cache from the given cache configuration.
   */
  private static CacheCreation generateDummyCache(String cacheConfig, 
                 DynamicRegionFactory.Config dynamicRegionConfig, String fn) {

    // open the DynamicRegionFactory, if any
    // before creating the cache
    if (dynamicRegionConfig != null) {
      log("Opening the configured DynamicRegionFactory");
      DynamicRegionFactory.get().open(dynamicRegionConfig);
    }

    // create and configure the dummy cache
    log("Creating dummy cache from config: " + cacheConfig);
    CacheDescription cd = getCacheDescription(cacheConfig);
    CacheCreation dummyCache = new CacheCreation();
    CacheVersionHelper.generateDummyResourceManager(cd, dummyCache);
    cd.configureDummy(dummyCache);

    // add the dynamic region factory configuration, if any
    if (dynamicRegionConfig != null) {
      dummyCache.setDynamicRegionFactoryConfig(dynamicRegionConfig);
    }

    // save the cache config for future reference
    XmlCacheConfigs.put(fn, cacheConfig);
    log("Created dummy cache: " + cacheToString(dummyCache));
    return dummyCache;
  }

  /**
   * Generates a dummy GatewayHub (and underlying Gateways) from the given cache and gateway configurations
   */
  private static void generateDummyGatewayHub(Cache dummyCache, String gatewayHubConfig, String fn) {

    if (gatewayHubConfig != null) {
      // create and configure the dummy gatewayHub
      log("Adding dummy gateway hub from config: " + gatewayHubConfig);
      GatewayHubDescription ghd = GatewayHubHelper.getGatewayHubDescription(gatewayHubConfig);

      String id = GatewayHubHelper.getId(ghd);
      int port = GatewayHubHelper.getPort(ghd, id);
      GatewayHub dummyGatewayHub = dummyCache.addGatewayHub(id, port);
      ghd.configure(dummyGatewayHub);
      log("Added dummy gateway hub: " + GatewayHubHelper.gatewayHubToString(dummyGatewayHub));

      // save the gateway hub config for future reference
      XmlGatewayHubConfigs.put(fn, gatewayHubConfig);
    }
  }


  /**
   * Generates a dummy Gateway from the given cache and gatewayHub configurations
   */
  private static void generateDummyGateways(Cache dummyCache, String gatewayConfig, String fn) {

    for (Iterator i = dummyCache.getGatewayHubs().iterator(); i.hasNext(); ) {
      GatewayHub dummyGatewayHub = (GatewayHub)i.next();
      GatewayDescription gd = GatewayHelper.getGatewayDescription(gatewayConfig);
      List endpoints = GatewayHubHelper.getEndpoints();
      String ownds = DistributedSystemHelper.getDistributedSystemName();
      Set dsnames = GatewayHubHelper.getDistributedSystemNames(endpoints);
      for (Iterator j = dsnames.iterator(); j.hasNext();) {
        String dsname = (String)j.next();
        if (!dsname.equals(ownds)) {
          int vmid = RemoteTestModule.getMyVmid();
          String gid = dsname; // remote distributed system
  
          log("Configuring gateway with id: " + gid);
          log("Adding dummy gateway from config: " + gatewayConfig);
          Gateway dummyGateway = dummyGatewayHub.addGateway(gid);
          gd.configure(dummyGateway, dsname, endpoints, true);
          XmlGatewayConfigs.put(fn, gatewayConfig);
          log("Configured gateway: " + GatewayHelper.gatewayToString(dummyGateway));
        }
      }
    }
  }

  /**
   * Generates a dummy pool from the given cache and pool configuration.
   */
  private static void generateDummyPool(Cache dummyCache, String poolConfig,
                                                          String fn) {
    if (poolConfig != null) {
      // create and configure the dummy pool
      log("Adding dummy pool from config: " + poolConfig);
      PoolDescription pd = PoolHelper.getPoolDescription(poolConfig);
      PoolFactory dummyFactory =
        ((CacheCreation)dummyCache).createPoolFactory();
      pd.configure(dummyFactory);
      String poolName = pd.getName();
      Pool dummyPool = null;
      try {
        dummyPool = dummyFactory.create(poolName);
      } catch (IllegalStateException e) {
        String msg = "A pool named \"" + poolName + "\" already exists.";
        if (e.getMessage().equals(msg)) {
          if (!XmlPoolConfigs.values().contains(poolConfig)) {
            String s = "Pool with configuration " + poolConfig
                     + " named " + poolName
                     + " was already created without CacheHelper XML using"
                     + " an unknown, and possibly different, configuration";
            throw new HydraRuntimeException(s);
          } // else the configuration was done here, which is fine
        } else {
          throw e;
        }
      }
      log("Added dummy pool: " + PoolHelper.poolToString(dummyPool));

      // save the pool config for future reference
      XmlPoolConfigs.put(fn, poolConfig);
    }
  }

  /**
   * Generates a dummy disk store from the given cache and disk store
   * configuration.
   */
  private static void generateDummyDiskStore(Cache dummyCache,
                                   String diskStoreConfig, String fn) {
    if (diskStoreConfig != null) {
      // create and configure the dummy disk store
      log("Adding dummy disk store from config: " + diskStoreConfig);
      CacheVersionHelper.generateDummyDiskStore(dummyCache, diskStoreConfig,
                                                fn, XmlDiskStoreConfigs);
    }
  }

  /**
   * Generates a dummy gateway sender from the given cache and gateway
   * sender configuration.
   */
  private static void generateDummyGatewaySender(Cache dummyCache,
                                   String gatewaySenderConfig, String fn) {
    if (gatewaySenderConfig != null) {
      // create and configure the dummy gateway sender
      log("Adding dummy gateway sender from config: " + gatewaySenderConfig);
      CacheVersionHelper.generateDummyGatewaySender(dummyCache,
                         gatewaySenderConfig, fn, XmlGatewaySenderConfigs);
    }
  }

  /**
   * Generates a dummy gateway receiver from the given cache and gateway
   * receiver configuration.
   */
  private static void generateDummyGatewayReceiver(Cache dummyCache,
                                   String gatewayReceiverConfig, String fn) {
    if (gatewayReceiverConfig != null) {
      // create and configure the dummy gateway receiver
      log("Adding dummy gateway receiver from config: " + gatewayReceiverConfig);
      CacheVersionHelper.generateDummyGatewayReceiver(dummyCache,
                         gatewayReceiverConfig, fn, XmlGatewayReceiverConfigs);
    }
  }

  /**
   * Generates a dummy async event queue from the given cache and async
   * event queue configuration.
   */
  private static void generateDummyAsyncEventQueue(Cache dummyCache,
                                   String asyncEventQueueConfig, String fn) {
    if (asyncEventQueueConfig != null) {
      // create and configure the dummy async event queue
      log("Adding dummy async event queue from config: " + asyncEventQueueConfig);
      CacheVersionHelper.generateDummyAsyncEventQueue(dummyCache,
                         asyncEventQueueConfig, fn, XmlAsyncEventQueueConfigs);
    }
  }

  /**
   * Generates a dummy bridge from the given cache and bridge configuration.
   */
  private static void generateDummyBridge(Cache dummyCache,
                                          String bridgeConfig, String fn) {
    if (bridgeConfig != null) {
      // create and configure the dummy bridge server
      log("Adding dummy bridge server from config: " + bridgeConfig);
      BridgeDescription bd = BridgeHelper.getBridgeDescription(bridgeConfig);
      CacheServer dummyCacheServer = dummyCache.addCacheServer();
      bd.configure(dummyCacheServer, BridgeHelper.getEndpoint(bd.getGroups())
                                                 .getPort());
      log("Added dummy bridge server: "
         + BridgeHelper.bridgeServerToString(dummyCacheServer));

      // save the bridge config for future reference
      XmlBridgeConfigs.put(fn, bridgeConfig);
    }
  }

  /**
   * Generates a dummy region from the given cache, region configuration, and
   * region name.
   */
  private static void generateDummyRegion(Cache dummyCache, String regionConfig,
                                          String regionName, String fn) {
    if (regionConfig != null) {
      // create and configure the dummy region
      log("Creating dummy region attributes from config: " + regionConfig);
      RegionDescription rd = RegionHelper.getRegionDescription(regionConfig);
      AttributesFactory regFactory = new AttributesFactory();
      rd.configure(regionName, regFactory, false);
      RegionAttributes dummyRatts =
        CacheVersionHelper.getDummyRegionAttributes(dummyCache,
                                                    regFactory.create());
      String rName = regionName;
      if (rName == null) {
        rName = RegionHelper.getRegionDescription(regionConfig).getRegionName();
      }
      log("Creating dummy region named: " + rName + " with attributes: "
         + RegionHelper.regionAttributesToString(dummyRatts));
      try {
        Region region = dummyCache.createRegion(rName, dummyRatts);
        log("Created dummy region named: " + rName);
      } catch (RegionExistsException e) {
        throw new HydraInternalException("Should not happen", e);
      }

      // save the region config for future reference
      XmlRegionConfigs.put(fn, regionConfig);
    }
  }

  /**
   * Generates dummy functions from the given cache and function list.
   */
  private static void generateDummyFunctions(Cache dummyCache, List functions,
                                                               String fn) {
    if (functions != null) {
      // create and configure the dummy functions
      log("Adding dummy functions: " + functions);
      FunctionServiceCreation fsc = new FunctionServiceCreation();
      for (Iterator i = functions.iterator(); i.hasNext();) {
        Function function = (Function)i.next();
        fsc.registerFunction(function);
      }
      //fsc.create(); // not needed, functions are registered during createCacheWithXml
      ((CacheCreation)dummyCache).setFunctionServiceCreation(fsc);
      log("Added dummy functions: " + fsc.getFunctions());

      // save the functions for future reference
      XmlFunctionConfigs.put(fn, classnamesFor(functions));
    }
  }
  private static List classnamesFor(List objs) {
    List classnames = null;
    if (objs != null) {
      classnames = new ArrayList();
      for (Iterator i = objs.iterator(); i.hasNext();) {
        Object obj = i.next();
        classnames.add(obj.getClass().getName());
      }
    }
    return classnames;
  }

  /**
   * Generates a cache XML file with the given filename.  Overwrites any
   * existing file with the same name.  This method is not thread safe.
   */
  private static void generateCacheXmlFile(String fn, Cache dummyCache) {
    log("Generating XML file: " + fn);
    PrintWriter pw = null;
    try {
      pw = new PrintWriter(new FileWriter(new File(fn)));
    } catch (IOException e) {
      String s = "Unable to open file: " + fn;
      throw new HydraRuntimeException(s, e);
    }
    CacheXmlGenerator.generate(dummyCache, pw);
    log("Generated XML file: " + fn);
  }

  /**
   * Returns the cache if it exists and is open, or null if no cache exists.
   */
  public static synchronized Cache getCache() {
    return CacheVersionHelper.getCache();
  }

  /**
   * Closes the cache if it exists and is open.
   */
  public static synchronized void closeCache() {
    Cache cache = getCache();
    if (cache != null) {
      log("Closing cache: " + cacheToString(cache));
      cache.close();
      log("Closed cache");
      TheCacheConfig = null; // so the next create can have a different config
    }
  }

  /**
   * Returns the given cache as a string.
   */
  public static String cacheToString(Cache cache) {
    return CacheDescription.cacheToString(cache);
  }

//------------------------------------------------------------------------------
// CacheDescription
//------------------------------------------------------------------------------

  /**
   * Returns the {@link CacheDescription} with the given configuration name
   * from {@link CachePrms#names}.
   */
  public static CacheDescription getCacheDescription(String cacheConfig) {
    if (cacheConfig == null) {
      throw new IllegalArgumentException("cacheConfig cannot be null");
    }
    log("Looking up cache config: " + cacheConfig);
    CacheDescription cd = TestConfig.getInstance()
                                    .getCacheDescription(cacheConfig);
    if (cd == null) {
      String s = cacheConfig + " not found in "
               + BasePrms.nameForKey(CachePrms.names);
      throw new HydraRuntimeException(s);
    }
    log("Looked up cache config:\n" + cd);
    return cd;
  }

//------------------------------------------------------------------------------
// Log
//------------------------------------------------------------------------------

  private static LogWriter log;
  private static synchronized void log(String s) {
    if (log == null) {
      log = Log.getLogWriter();
    }
    if (log.infoEnabled()) {
      log.info(s);
    }
  }
}
