package io.smallrye.context;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.microprofile.context.ManagedExecutor;
import org.eclipse.microprofile.context.ThreadContext;
import org.eclipse.microprofile.context.spi.ContextManager;
import org.eclipse.microprofile.context.spi.ContextManagerExtension;
import org.eclipse.microprofile.context.spi.ThreadContextProvider;

import io.smallrye.context.impl.DefaultValues;
import io.smallrye.context.impl.ManagedExecutorBuilderImpl;
import io.smallrye.context.impl.ThreadContextBuilderImpl;
import io.smallrye.context.impl.ThreadContextProviderPlan;

public class SmallRyeContextManager implements ContextManager {

    public static final String[] NO_STRING = new String[0];

    public static final String[] ALL_REMAINING_ARRAY = new String[] { ThreadContext.ALL_REMAINING };

    private List<ThreadContextProvider> providers;
    private List<ContextManagerExtension> extensions;
    private Map<String, ThreadContextProvider> providersByType;

    private String[] allProviderTypes;

    private DefaultValues defaultValues;

    SmallRyeContextManager(List<ThreadContextProvider> providers, List<ContextManagerExtension> extensions) {
        this.providers = new ArrayList<ThreadContextProvider>(providers);
        providersByType = new HashMap<>();
        for (ThreadContextProvider provider : providers) {
            providersByType.put(provider.getThreadContextType(), provider);
        }
        // FIXME: check for duplicate types
        // FIXME: check for cycles
        allProviderTypes = providersByType.keySet().toArray(new String[this.providers.size()]);
        this.extensions = new ArrayList<ContextManagerExtension>(extensions);
        this.defaultValues = new DefaultValues();
        // Extensions may call our methods, so do all init before we call this
        for (ContextManagerExtension extension : extensions) {
            extension.setup(this);
        }
    }

    public String[] getAllProviderTypes() {
        return allProviderTypes;
    }

    public CapturedContextState captureContext(ThreadContextProviderPlan plan) {
        Map<String, String> props = Collections.emptyMap();
        return new CapturedContextState(this, plan, props);
    }

    // for tests
    public ThreadContextProviderPlan getProviderPlan() {
        return getProviderPlan(allProviderTypes, NO_STRING, NO_STRING);
    }

    public ThreadContextProviderPlan getProviderPlan(String[] propagated, String[] unchanged, String[] cleared) {
        Set<String> propagatedSet = new HashSet<>();
        Collections.addAll(propagatedSet, propagated);

        Set<String> clearedSet = new HashSet<>();
        Collections.addAll(clearedSet, cleared);

        Set<String> unchangedSet = new HashSet<>();
        Collections.addAll(unchangedSet, unchanged);

        // check for duplicates
        if (propagatedSet.removeAll(unchangedSet) || propagatedSet.removeAll(clearedSet)
                || clearedSet.removeAll(propagatedSet) || clearedSet.removeAll(unchangedSet)
                || unchangedSet.removeAll(propagatedSet) || unchangedSet.removeAll(clearedSet)) {
            throw new IllegalStateException(
                    "Cannot use ALL_REMAINING in more than one of propagated, cleared, unchanged");
        }

        // expand ALL_REMAINING
        boolean hadAllRemaining = false;
        if (propagatedSet.contains(ThreadContext.ALL_REMAINING)) {
            propagatedSet.remove(ThreadContext.ALL_REMAINING);
            Collections.addAll(propagatedSet, allProviderTypes);
            propagatedSet.removeAll(clearedSet);
            propagatedSet.removeAll(unchangedSet);
            hadAllRemaining = true;
        }

        if (unchangedSet.contains(ThreadContext.ALL_REMAINING)) {
            unchangedSet.remove(ThreadContext.ALL_REMAINING);
            Collections.addAll(unchangedSet, allProviderTypes);
            unchangedSet.removeAll(propagatedSet);
            unchangedSet.removeAll(clearedSet);
            hadAllRemaining = true;
        }

        // cleared implicitly defaults to ALL_REMAINING if nobody else is using it
        if (clearedSet.contains(ThreadContext.ALL_REMAINING) || !hadAllRemaining) {
            clearedSet.remove(ThreadContext.ALL_REMAINING);
            Collections.addAll(clearedSet, allProviderTypes);
            clearedSet.removeAll(propagatedSet);
            clearedSet.removeAll(unchangedSet);
        }

        // check for existence
        Set<ThreadContextProvider> propagatedProviders = new HashSet<>();
        for (String type : propagatedSet) {
            if (type.isEmpty()) {
                continue;
            }
            ThreadContextProvider provider = providersByType.get(type);
            if (provider == null)
                throw new IllegalStateException("Missing propagated provider type: " + type);
            propagatedProviders.add(provider);
        }

        // ignore missing for cleared/unchanged
        Set<ThreadContextProvider> unchangedProviders = new HashSet<>();
        for (String type : unchangedSet) {
            if (type.isEmpty()) {
                continue;
            }
            ThreadContextProvider provider = providersByType.get(type);
            if (provider != null)
                unchangedProviders.add(provider);
        }

        Set<ThreadContextProvider> clearedProviders = new HashSet<>();
        for (String type : clearedSet) {
            if (type.isEmpty()) {
                continue;
            }
            ThreadContextProvider provider = providersByType.get(type);
            if (provider != null)
                clearedProviders.add(provider);
        }

        return new ThreadContextProviderPlan(propagatedProviders, unchangedProviders, clearedProviders);
    }

    @Override
    public ManagedExecutor.Builder newManagedExecutorBuilder() {
        return new ManagedExecutorBuilderImpl(this);
    }

    @Override
    public ThreadContext.Builder newThreadContextBuilder() {
        return new ThreadContextBuilderImpl(this);
    }

    // For tests
    public List<ContextManagerExtension> getExtensions() {
        return extensions;
    }

    public DefaultValues getDefaultValues() {
        return defaultValues;
    }
}