package com.shipsy.ondemand.riderapp.framework.network

import com.shipsy.finance.constants.GET_FINANCE_SCREEN_DATA
import com.shipsy.finance.constants.GET_RIDER_PAYOUT
import com.shipsy.finance.models.network.FinanceResponse
import com.shipsy.finance.models.network.PayoutData
import com.shipsy.ondemand.core.framework.util.getRequest
import com.shipsy.ondemand.core.framework.util.postRequest
import com.shipsy.ondemand.riderapp.framework.data.codsettlement.CashSettlementListRequest
import com.shipsy.ondemand.riderapp.framework.data.codsettlement.CashSettlementRequest
import com.shipsy.ondemand.riderapp.framework.data.codsettlement.SettlementData
import com.shipsy.ondemand.riderapp.framework.data.loginhistory.LoginHistoryData
import com.shipsy.ondemand.riderapp.framework.data.orderhistory.OrderHistoryData
import com.shipsy.ondemand.riderapp.framework.data.sdk.SDKMasterDataFetchResponse
import com.shipsy.ondemand.riderapp.framework.data.sdk.SdkAppSettings
import com.shipsy.ondemand.riderapp.framework.network.mapper.mapLoginResponseToLoginResponse
import com.shipsy.ondemand.riderapp.framework.network.mapper.mapOrgResponseToOrgOutput
import com.shipsy.ondemand.riderapp.framework.network.model.*
import com.shipsy.ondemand.riderapp.framework.network.model.checkin.CheckInRequest
import com.shipsy.ondemand.riderapp.framework.network.model.checkout.CheckoutRequest
import com.shipsy.ondemand.riderapp.framework.network.model.codsettlement.CreateQrCodeRequestBody
import com.shipsy.ondemand.riderapp.framework.network.model.codsettlement.CreateQrCodeResponse
import com.shipsy.ondemand.riderapp.framework.network.model.file.ImageTypeModel
import com.shipsy.ondemand.riderapp.framework.network.model.file.UploadDocumentsData
import com.shipsy.ondemand.riderapp.framework.network.model.login.AppConfigurationData
import com.shipsy.ondemand.riderapp.framework.network.model.login.AppUpdate
import com.shipsy.ondemand.riderapp.framework.network.model.login.LoginResponse
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.order.UpcomingOrdersModel
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderStateRequest
import com.shipsy.ondemand.riderapp.framework.network.model.transactionhistory.TransactionData
import com.shipsy.ondemand.riderapp.framework.network.model.transactionhistory.TransactionHistoryRequestBody
import com.shipsy.ondemand.riderapp.interactor.NetworkClient
import com.shipsy.ondemand.riderapp.interactor.data.OrgLoginResponse
import com.shipsy.ondemand.riderapp.interactor.usecase.nativeAnalytics.AnalyticsNativeUseCase
import com.shipsy.ondemand.riderapp.models.CallStatusRequest
import com.shipsy.ondemand.riderapp.models.CallStatusUpdateResponse
import io.ktor.client.*
import io.ktor.client.request.*
import io.ktor.http.*

class NetworkClientImpl(private val client: HttpClient) :
    NetworkClient {
    override suspend fun getOrgDetails(orgName: String): OrgLoginResponse {
        return client.getRequest<BaseResponse<OrganisationData>>(
            OrgRequestURL,
            headers = mapOf("organisation-pretty-name" to orgName)
        ).mapOrgResponseToOrgOutput()
    }

    override suspend fun login(user: UserLogin): LoginResponse {
        return client.postRequest<UserLogin, BaseResponse<LoginResponse>>(
            LoginRequestURL,
            user
        ).mapLoginResponseToLoginResponse()
    }

    override suspend fun checkIn(
        workerId: String,
        checkInRequest: CheckInRequest
    ): RiderStateResponse? {
        return client.postRequest(
            CheckInRequestURL,
            query = mapOf("worker_id" to workerId),
            body = checkInRequest
        )
    }

    override suspend fun fetchRiderState(): RiderStateResponse {
        return client.getRequest(
            RiderStateURL
        )
    }

    override suspend fun postRiderEvents(riderEvent: RiderStateRequest): RiderStateResponse {
        return client.postRequest(
            RiderStateURL,
            body = riderEvent
        )
    }

    override suspend fun checkoutRider(
        workerId: String,
        checkoutRequest: CheckoutRequest
    ): BaseResponse<Boolean?> {
        return client.postRequest(
            CheckoutURL,
            query = mapOf("worker_id" to workerId),
            body = checkoutRequest
        )
    }

    override suspend fun fetchUpcomingOrders(hubCode: String): BaseResponse<UpcomingOrdersModel?> {
        return client.getRequest(
            UpcomingOrderURL,
            query = mapOf("hub_code" to hubCode)
        )
    }

    override suspend fun logoutUser(): BaseResponse<Boolean?> {
        return client.postRequest<Any, BaseResponse<Boolean?>>(
            LogoutURL
        )
    }

    override suspend fun getOrderHistory(
        fromDate: String,
        toDate: String
    ): BaseResponse<OrderHistoryData> {
        return client.getRequest(
            GET_ALL_PAST_ORDERS,
            query = mapOf("from_date" to fromDate, "to_date" to toDate)
        )
    }

    override suspend fun fetchAppSetting(workerId: String): BaseResponse<AppConfigurationData> {
        return client.getRequest(
            AppSettingsURL,
            query = mapOf("worker_id" to workerId)
        )
    }

    override suspend fun getUrlForStorage(model: ImageTypeModel): BaseResponse<UploadDocumentsData> {
        return client.postRequest(UPLOAD_DOCUMENTS, body = model)
    }

    override suspend fun uploadFile(uploadUrl: String, bytes: ByteArray) {
        client.put(uploadUrl) {
            contentType(ContentType.Image.JPEG)
            setBody(bytes)
        }
    }

    override suspend fun getLoginHistory(ofDate: String): BaseResponse<LoginHistoryData> {
        return client.getRequest(GET_LOGIN_HOURS, query = mapOf("date" to ofDate))
    }

    override suspend fun fetchCodTasks(body: CashSettlementListRequest): BaseResponse<SettlementData> {
        return client.postRequest(GET_CASH_SETTLEMENT_LIST, body = body)
    }

    override suspend fun postCashSettlementRequest(body: List<CashSettlementRequest>): BaseResponse<Boolean?> {
        return client.postRequest(POST_CASH_SETTLEMENT_REQUEST, body = body)
    }

    override suspend fun createQrCode(body: CreateQrCodeRequestBody): BaseResponse<CreateQrCodeResponse> {
        return client.postRequest(CREATE_QR_CODE, body = body)
    }

    override suspend fun fetchRejectedRequests(body: TransactionHistoryRequestBody): BaseResponse<TransactionData> {
        return client.postRequest(GET_REJECTED_SETTLEMENT_LIST, body = body)
    }

    override suspend fun fetchApprovedRequests(body: TransactionHistoryRequestBody): BaseResponse<TransactionData> {
        return client.postRequest(GET_APPROVED_SETTLEMENT_LIST, body = body)
    }

    override suspend fun fetchRaisedRequests(body: TransactionHistoryRequestBody): BaseResponse<TransactionData> {
        return client.postRequest(GET_RAISED_SETTLEMENT_LIST, body = body)
    }

    override suspend fun postWithdrawRequest(body: WithdrawRequestModel): BaseResponse<Boolean?> {
        return client.postRequest(POST_WITHDRAW_REQUEST, body = body)
    }

    override suspend fun fcmTokenUpdate(token: String): BaseResponse<Boolean?> {
        return client.postRequest<Any, BaseResponse<Boolean?>>(
            UpdateFcmTokenURL,
            body = FCMTokenRequest(token)
        )
    }

    override suspend fun sdkMasterDataFetch(): BaseResponse<SDKMasterDataFetchResponse> {
        return client.getRequest(FETCH_SDK_MASTER_DATA)
    }

    override suspend fun checkSdkHealth(): BaseResponse<SdkAppSettings> {
        return client.getRequest(CHECK_SDK_HEALTH)
    }

    override suspend fun updateCallReasonSelection(body: CallStatusRequest): BaseResponse<CallStatusUpdateResponse> {
        return client.postRequest(CALL_STATUS_OPTIONS, body = body)
    }

    override suspend fun checkForUpdate(workerId: String): BaseResponse<AppUpdate> {
        return client.getRequest(GET_APP_UPDATE_API)
    }

    override suspend fun getFinanceScreenData(
        date : String
    ): BaseResponse<FinanceResponse> {
        return client.getRequest(
            GET_FINANCE_SCREEN_DATA,
            query = mapOf("date" to date)
        )
    }

    override suspend fun getRiderPayout(
        month: String,
        year: String,
        week: String,
        day: String,
        referenceNumber: String
    ): BaseResponse<PayoutData?> {
        return client.getRequest(
            GET_RIDER_PAYOUT,
            query = mapOf(
                "month" to month,
                "year" to year,
                "week" to week,
                "day" to day,
                "referenceNumber" to referenceNumber
            )
        )
    }
}