package com.shipsy.ondemand.riderapp.framework.data.loginhistory

import com.shipsy.ondemand.riderapp.domain.const.BREAK_HOUR_TYPE
import com.shipsy.ondemand.riderapp.domain.const.LOGIN_HOUR_TYPE
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

typealias LoginHistoryData = List<LoginHistoryItem?>?

const val SECONDS_IN_MINUTE = 60
const val MINUTES_IN_HOUR = 60
const val SECONDS_IN_HOUR = SECONDS_IN_MINUTE * MINUTES_IN_HOUR

@Serializable
data class LoginHistoryItem(
    @SerialName("week") val week: Int?,
    @SerialName("duration") val duration: List<DurationData?>?,
    @SerialName("data") val data: List<DayData?>?
) {
    val durationDisplayText: String
        get() = duration.displayText()
}

@Serializable
data class DayData(
    @SerialName("date") val date: String?,
    @SerialName("day") val day: String?,
    @SerialName("duration") val duration: List<DurationData?>?,
    @SerialName("data") val data: List<HourData?>?
) {
    val hasWorkHours: Boolean
        get() = !workHourData?.data.isNullOrEmpty()

    val workHours: DurationData?
        get() = duration?.filterNotNull()?.firstOrNull { it.isTypeLogin }

    val workHourData: HourData?
        get() = data?.filterNotNull()?.firstOrNull { it.isTypeLogin }

    val hasBreakHours: Boolean
        get() = !breakHourData?.data.isNullOrEmpty()

    val breakHours: DurationData?
        get() = duration?.filterNotNull()?.firstOrNull { it.isTypeBreak }

    val breakHourData: HourData?
        get() = data?.filterNotNull()?.firstOrNull { it.isTypeBreak }
}

@Serializable
data class DurationData(
    @SerialName("type") val type: String?,
    @SerialName("duration") val duration: Long?
) {
    val isTypeLogin
        get() = (type == LOGIN_HOUR_TYPE)
    val isTypeBreak
        get() = (type == BREAK_HOUR_TYPE)
    val durationDisplayText: String
        get() = duration.durationDisplayText()
}

@Serializable
data class HourData(
    @SerialName("type") val type: String?,
    @SerialName("data") val data: List<SlotData?>?
) {
    val isTypeLogin
        get() = (type == LOGIN_HOUR_TYPE)
    val isTypeBreak
        get() = (type == BREAK_HOUR_TYPE)
}

@Serializable
data class SlotData(
    @SerialName("slot") val slot: String?,
    @SerialName("duration") val duration: Long?
) {
    val durationDisplayText: String
        get() = duration.durationDisplayText()
}

private fun List<DurationData?>?.displayText(): String {
    if (this == null) return ""
    if (filterNotNull().isEmpty()) return ""

    val workHours = hoursText("Work Hours") { it.isTypeLogin }
    val breakHours = hoursText("Break Hours") { it.isTypeBreak }

    return listOfNotNull(workHours, breakHours).joinToString(separator = " | ")
}

private fun List<DurationData?>.hoursText(suffix: String, test: (DurationData) -> Boolean): String? {
    return filterNotNull()
        .filter { it.duration != null && it.duration > 0 }
        .firstOrNull { test(it) }
        ?.let {
            val hours = it.duration!! / SECONDS_IN_HOUR
            if (hours == 0L) null else "$hours $suffix"
        }
}


private fun Long?.durationDisplayText(): String {
    if (this == null) return ""
    var totalSeconds = this.toInt()
    val hrs: Int = totalSeconds / SECONDS_IN_HOUR
    totalSeconds -= hrs * SECONDS_IN_HOUR
    val mins: Int = totalSeconds / SECONDS_IN_MINUTE
    totalSeconds -= mins * SECONDS_IN_MINUTE
    val s = StringBuilder()
    if (hrs != 0) {
        s.append(hrs).append(" h ")
    }
    if (mins != 0) s.append(mins).append(" m ")
    if (totalSeconds != 0) s.append(totalSeconds).append(" s ")
    return s.toString()
}