package com.shipsy.ondemand.riderapp.firebase

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.framework.data.oderdata.OrderData
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.interactor.usecase.SetFirebaseEventsUserProperties
import com.shipsy.ondemand.riderapp.interactor.usecase.nativeAnalytics.AnalyticsNativeUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.FetchRiderStateFromLocal
import org.koin.mp.KoinPlatformTools

/**
 * Created by Kalpesh Kundanani on 06/02/23
 */
object FirebaseEventHandler {
    private const val TAG = "FirebaseEventHandler"
    private var sessionTicket: String? = ""
    private const val userPropertyUpdated = "USER_PROPERTY_UPDATED"

    private val localStore: LocalStore
        get() {
            val store by lazy { KoinPlatformTools.defaultContext().get().get<LocalStore>() }
            return store
        }
    
    private val analyticsNativeUseCase: AnalyticsNativeUseCase
        get() {
            val nativeUseCase by lazy { KoinPlatformTools.defaultContext().get().get<AnalyticsNativeUseCase>() }
            return nativeUseCase
        }

    private val setFirebaseEventProperties: SetFirebaseEventsUserProperties
        get() {
            val nativeUseCase by lazy { KoinPlatformTools.defaultContext().get().get<SetFirebaseEventsUserProperties>() }
            return nativeUseCase
        }

    private val fetchRiderStateFromLocal: FetchRiderStateFromLocal
        get() {
            val nativeUseCase by lazy { KoinPlatformTools.defaultContext().get().get<FetchRiderStateFromLocal>() }
            return nativeUseCase
        }

    private fun generateSessionTicket(): String {
        val code = localStore.getValue(worker_code, "before")
        val org = localStore.getValue(ORG_ID, "login")
        return org + "_" + code + "_" + DateTime.currentTimeMillis()
    }

    private fun updateSessionTicket() {
        sessionTicket = generateSessionTicket()
        print("sessionTicket: $sessionTicket")
    }

    fun initializeFirebase() {
        updateSessionTicket()
    }

    fun logFirebaseEvent(event: String, parameters: Map<String, Any?>? = null) {
        println("$TAG: logFirebaseEvent: $event data $parameters")
        val input = AnalyticsNativeUseCase.Input(event, parameters)
        analyticsNativeUseCase.invoke(input)
    }

    fun logLocationServiceStart(startedBy: String) {
        val map = mapOf("started_by" to startedBy)
        logFirebaseEvent(locationServiceStart, map)
    }

    fun logLocationServiceStop(stopBy: String) {
        val map = mapOf("stop_by" to stopBy)
        logFirebaseEvent(locationServiceStop, map)
    }

    fun logLocationServiceError() {
        logFirebaseEvent(locationTrackingErrorEvent)
    }

    fun logUpdateFirebaseAuthEvent(isFirebaseUserLoggedIn: Boolean, token: String?) {
        val bundle = mapOf(
            "is_firebase_user_logged_in" to isFirebaseUserLoggedIn,
            "firebase_token_length" to (token?.length ?: 0)
        )
        this.logFirebaseEvent(fireBaseAuthDetailsEvent, bundle)
    }

    fun logFirebaseAuthSignInEvent(loginSuccess: Boolean, exception: String?) {
        val bundle = mapOf("exception" to exception)
        val status = if (loginSuccess) "success" else "failure"
        val eventName = firebaseAuthSigninEvent.replace("{status}", status)
        logFirebaseEvent(eventName, bundle)
    }

    fun logButtonClickEvent(btText: String) {
        val eventName = buttonClick.replace("{text}", btText.lowercase().replace(" ", "_"))
        logFirebaseEvent(eventName)
    }

    fun logAppState(state: String) {
        val eventName = appState.replace("{state}", state.lowercase().replace(" ", "_"))
        logFirebaseEvent(eventName)
    }

    fun logShowPopUpEvent(tittle: String) {
        val eventName = displayPopup.replace("{tittle}", tittle.lowercase().replace(" ", "_"))
        logFirebaseEvent(eventName)
    }

    fun logScreenLoadEvent(screenName: String) {
        val eventName = screenLoad.replace("{screen}", screenName.lowercase())
        logFirebaseEvent(eventName)
    }

    fun logButtonSwipeEvent(slideEvent: String, autoSwipe: Boolean = false) {
        val eventName = buttonSwipe.replace("{text}", slideEvent.lowercase().replace(" ", "_"))
        val bundle = mapOf("auto_swipe" to autoSwipe)
        this.logFirebaseEvent(eventName, bundle)
    }

    fun logMockLocationEvent(isMock: Boolean) {
        val eventName = mockLocation.replace("{isMock}", "$isMock")
        logFirebaseEvent(eventName)
    }

    fun logOrderStateEvent(event: String, referenceNumber: String) {
        val eventName = orderStateEvent.replace("{state}", event)
        val data = mapOf("reference_number" to referenceNumber)
        this.logFirebaseEvent(eventName, data)
    }

    fun logRiderStateEvent(event: String, referenceNumber: List<String>) {
        val eventName = riderStateEvent.replace("{state}", event)
        if (referenceNumber.isNotEmpty())
            for (refNum in referenceNumber) {
                val data = mapOf("reference_number" to refNum)
                this.logFirebaseEvent(eventName, data)
            }
        else
            logFirebaseEvent(eventName)

    }

    fun logFunctionEvent(event: String, data: Map<String, Any?>? = null) {
        val eventName = functionalEvent.replace("{text}", event)
        this.logFirebaseEvent(eventName, data)
    }


    fun logNavSelectedEvent(navItem: String) {
        val eventName = navSelection.replace("{item}", navItem.lowercase().replace(" ", "_"))
        logFirebaseEvent(eventName)

    }


    fun logFirebaseApiRequestEvent(apiUrl: String, eventData: Map<String, Any?>? = null) {
        val apiEvent = apiRequest.replace("{url}", urlExtraction(apiUrl).lowercase())
        this.logFirebaseEvent(apiEvent, eventData)
    }

    fun logFirebaseApiResponseEvent(
        apiUrl: String,
        code: Int,
        method: String,
        responseTime: Long,
        eventData: HashMap<String, Any?>? = null
    ) {
        val apiEvent = apiResponse.replace("{url}", urlExtraction(apiUrl).lowercase())
            .replace("{method}", method)
        var data = eventData
        if (data == null)
            data = HashMap()
        data["response_code"] = code
        data["response_time"] = responseTime
        logFirebaseEvent(apiEvent, data)
    }

    private fun isUserPropertyUpdated(): Boolean {
        return localStore.getValue(userPropertyUpdated, false)
    }

    fun setUserPropertyUpdated(isUpdated: Boolean) {
        localStore.putValue(userPropertyUpdated, isUpdated)
    }

    fun updateUserDetailAfterLogin() {
        try {
            if (isUserPropertyUpdated())
                return

            val userProperty = HashMap<String, String>()
            userProperty[riderName] = localStore.getValue(rider_name, "")
            userProperty[organisationIdProp] = localStore.getValue(ORG_ID, "")
            userProperty[workerCode] = localStore.getValue(worker_code, "")
            userProperty[hubCode] = localStore.getValue(HUB_CODE, "")
            userProperty[adsId] = localStore.getValue(SHARED_PREF_ADS_ID, "") ?: ""

            updateSessionTicket()

            val userUniqueId = localStore.getValue(rider_id, "")

            setFirebaseEventProperties.invoke(SetFirebaseEventsUserProperties.Input(
                userUniqueId,
                userProperty
            ))

            setUserPropertyUpdated(true)
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

    private fun urlExtraction(apiUrl: String): String {
        if (apiUrl.contains("RiderApp/"))
            return removeRequestData(apiUrl.split("RiderApp/")[1]).replace("/", "_")
        if (apiUrl.contains("s3.amazonaws.com/")) {
            return "image_upload"
        }
        return apiUrl
    }

    private fun removeRequestData(apiUrl: String): String {
        if (apiUrl.contains("?"))
            return apiUrl.split("?")[0]
        return apiUrl
    }

    /*
    * Notification Events
    *
    * */
    fun logNotificationReceived(tittle: String?, refNum: String?) {
        val data = mapOf(
            "tittle" to tittle,
            "reference_number" to refNum
        )
        this.logFirebaseEvent(notificationReceived, data)
    }

    fun logNotificationSoundPlayed(tittle: String, refNum: String?) {
        val data = mapOf(
            "tittle" to tittle,
            "reference_number" to refNum
        )
        this.logFirebaseEvent(notificationSoundPlayed, data)
    }


    /*
    * Functional firebase event for order load and screen change
    *
    * */
    fun logUiUpdateEvent(riderState: RiderStateResponse?) {
        updateEventForOrder(updateUiForOrders, riderState)
    }


    fun logLocalStateUpdate(riderState: RiderStateResponse?) {
        updateEventForOrder(updateLocalState, riderState)
    }

    private fun updateEventForOrder(eventName: String, riderState: RiderStateResponse?) {
        val orderBundle = getOrderBundleFromState(riderState)
        if (orderBundle.isNotEmpty())
            for (bundle in orderBundle) {
                logFunctionEvent(eventName, bundle)
            }
        else
            logFunctionEvent(eventName)
    }

    private fun getOrderBundleFromState(
        riderState: RiderStateResponse?
    ): List<Map<String, Any?>> {
        val listOfBundle = ArrayList<Map<String, Any?>>()
        if (riderState?.current_state?.current_orders != null)
            for (order in riderState.current_state.current_orders) {
                listOfBundle.add(
                    mapOf(
                        "reference_number" to order.reference_number,
                        "order_status" to order.status
                    )
                )
            }
        return listOfBundle.toList()
    }

    suspend fun logUiObservedEvent() {
        val riderState = fetchRiderStateFromLocal.invoke()
        updateEventForOrder(
            updateUiForOrdersObserve,
            riderState.getOrNull()
        )
    }

    suspend fun logUiObservedEventIgnored() {
        val riderState = fetchRiderStateFromLocal.invoke()
        updateEventForOrder(
            updateUiForOrdersObserveIgnore,
            riderState.getOrNull()
        )
    }

    fun logOrderDisplayedEvent(currentOrders: List<OrderData>?) {
        if (currentOrders != null)
            for (order in currentOrders) {
                val data = mapOf(
                    "reference_number" to order.referenceNumber,
                    "time_delay" to getDisplayDelayTime(order.assign_time)
                )
                logFunctionEvent(
                    orderDisplayOnScreen,
                    data
                )
            }

    }

    private fun getDisplayDelayTime(assignTime: Long?): Long {
        if (assignTime == null) {
            return 0
        }
        return DateTime.currentTimeMillis() - assignTime
    }

    fun logUiUpdatedEvent(screen: String?, orderStatus: String? = null, referenceNumber: String? = null) {
        val data = mapOf(
            "moved_to" to screen,
            "order_status" to orderStatus,
            "reference_number" to referenceNumber
        )

        logFirebaseEvent(updateUiDone, data)
    }
//    fun logUiUpdatedEvent(screen: String?, orders: List<CurrentOrders>) =
//        orders.forEach { logUiUpdatedEvent(screen, it.status, it.reference_number) }

    fun logUiUpdatedEvent(screen: String?, orders: List<OrderData>) =
        orders.forEach { logUiUpdatedEvent(screen, it.status, it.referenceNumber) }

    fun logCheckInEvent(lat: Double, lng: Double, distance: Double) {
        val bundle = mapOf(
            "location" to "lat $lat long $lng",
            "distance" to distance
        )
        logFunctionEvent(
            riderCheckInAttemptLocation,
            bundle
        )
    }

    fun logImageUploadEvent(eventName: String, imageName: String? = "", bundle: HashMap<String, Any?>? = null) {
        val event = imageEvent.replace("{task}", eventName)
        val data = bundle ?: hashMapOf()
        data["image"] = imageName?.let { removeExtraDetailImageName(it) }
        this.logFirebaseEvent(event, data)
    }

    private fun removeExtraDetailImageName(name: String): String? {
        return name.split('/').lastOrNull()
    }

//    @JvmStatic
//    fun logActivityResultEvent(eventName: String, activityResult: ActivityResult?=null){
//        val event = activityResultEvent.replace("{task}", eventName)
//        val bundle = Bundle()
//        bundle.putString("ACTIVITY_RESULT", activityResult.toString())
//        this.logFirebaseEvent(event, bundle)
//    }
}