package com.shipsy.ondemand.riderapp.domain.usecase.statemanagement

import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.const.WorkerStatusType
import com.shipsy.ondemand.riderapp.domain.mapper.mapToOrderStatus
import com.shipsy.ondemand.riderapp.interactor.usecase.OrderPriorityUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.statemanagement.OrderToShowForDetailUseCase
import com.shipsy.ondemand.riderapp.shared.cache.CurrentOrders

class OrderToShowForDetailUseCaseImpl(
    private val database: Database,
    private val priorityUseCase: OrderPriorityUseCase,
) :
    OrderToShowForDetailUseCase {
    override suspend fun invoke(input: OrderToShowForDetailUseCase.Input): OrderToShowForDetailUseCase.Output {
        val isPudoFlow = input.isPudoFlow
        val orders = database.currentOrdersDao.getCurrentOrders()
        val shouldShowOrderDetail = orders.size == 1 && input.workerStatus == WorkerStatusType.OUT_FOR_DELIVERY
        return if (shouldShowOrderDetail) {
            handleSingleOrder(orders)
        } else {
            handleMultipleOrders(orders, isPudoFlow)
        }
    }

    /**
     * In case when order list or ongoing list is shown and only one order is there, this will
     * open the order details for that order.
     */
    private fun handleSingleOrder(orders: List<CurrentOrders>) =
        OrderToShowForDetailUseCase.Output(orders.first().reference_number, true)

    /**
     * In case of multiple orders, when order details is to be shown, this is used to decide which
     * order should appear first. Also, if order details is not to be shown, this will return null.
     */
    private fun handleMultipleOrders(
        orders: List<CurrentOrders>,
        isPudoFlow: Boolean
    ): OrderToShowForDetailUseCase.Output {
        val order =
            orders.filter { it.status.mapToOrderStatus(isPudoFlow = isPudoFlow) != null }
                .filter { priorityUseCase.invoke(it.status.mapToOrderStatus(isPudoFlow = isPudoFlow)!!) != null }
                .minByOrNull { priorityUseCase.invoke(it.status.mapToOrderStatus(isPudoFlow = isPudoFlow)!!)!! }

        println("OrderToShowForDetailUseCase: ${order?.order_number} | ${order?.reference_number} | orders.size: ${orders.size}")

        return OrderToShowForDetailUseCase.Output(order?.reference_number)
    }
}