package com.shipsy.ondemand.riderapp.domain.usecase.statemanagement

import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.mapper.mapToOrderStatus
import com.shipsy.ondemand.riderapp.domain.orderstatus.OrderStatus
import com.shipsy.ondemand.riderapp.firebase.FirebaseEventHandler
import com.shipsy.ondemand.riderapp.framework.data.OrderDetailStatus
import com.shipsy.ondemand.riderapp.framework.data.oderdata.OrderData
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.pickupscan.IsPickingFlowEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.IsPudoFlowEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.deliveryscan.IsManualEntryAllowedDeliveryScanUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.deliveryscan.IsScannerEnabledInDeliveryScanUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.statemanagement.OrderDetailScreenUseCase

class OrderDetailScreenUseCaseImpl(
    private val database: Database,
    private val isPickingFlowEnabledUseCase: IsPickingFlowEnabledUseCase,
    private val isPudoFlowEnabledUseCase: IsPudoFlowEnabledUseCase,
    private val isScannerEnabledUseCase: IsScannerEnabledInDeliveryScanUseCase,
    private val isManualEntryEnabledUseCase: IsManualEntryAllowedDeliveryScanUseCase,
) : OrderDetailScreenUseCase {
    override suspend fun invoke(input: OrderDetailScreenUseCase.Input): OrderDetailStatus? {

        val isPickingFlow = isPickingFlowEnabledUseCase.invoke()
        val isPudoFlow = isPudoFlowEnabledUseCase.invoke()
        val isScannerEnabled = isScannerEnabledUseCase.invoke()
        val isManualEntryEnabled = isManualEntryEnabledUseCase.invoke()
        val canScanHu = isScannerEnabled && isManualEntryEnabled
        val currentOrder = database.currentOrdersDao.getOrdersByReferenceNumber(input.referenceNumber) ?: return null

        val orderStatus = input.orderStatus.mapToOrderStatus(input.requestOrderDetail, isPudoFlow)
        val orderData = OrderData.fromCurrentOrder(currentOrder)
        println("\n\n==> orderStatus: $orderStatus currentOrder: ${currentOrder.order_number} \n\n")
        val orderDetailStatus = if (orderStatus == null)
            null
        else
            when (orderStatus) {
                OrderStatus.PICKUP_AWAITED -> {
                    OrderDetailStatus.AcceptScreen(
                        orderData,
                        false
                    )
                }
                OrderStatus.REACHED_GATE, OrderStatus.RTO_REACHED_GATE -> {
                    OrderDetailStatus.DeliveryScreen(
                        orderData,
                        false
                    )
                }
                OrderStatus.ASSIGNED_FOR_DELIVERY -> {
                    if (orderData.isPickupTask) {
                        OrderDetailStatus.GoingForPickup(
                            orderData,
                            true
                        )
                    } else {
                        OrderDetailStatus.ReachGateScreen(
                            orderData,
                            true
                        )
                    }
                }
                OrderStatus.PICKED_UP -> {
                    if (isPickingFlow && canScanHu) {
                        OrderDetailStatus.PickingUpStarted(
                            orderData,
                            true
                        )
                    } else {
                        OrderDetailStatus.CompletePickup(
                            orderData,
                            true
                        )
                    }
                }
                OrderStatus.PICKUP_COMPLETED, OrderStatus.START_BIKE -> {
                    OrderDetailStatus.ReachGateScreen(
                        orderData,
                        true
                    )
                }
            }

        orderDetailStatus?.let { FirebaseEventHandler.logUiUpdatedEvent(
            "${it::class.simpleName}",
            orderData.status,
            orderData.referenceNumber
        ) }

        return orderDetailStatus
    }
}