package com.shipsy.ondemand.riderapp.domain.usecase.sdk

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.framework.data.sdk.SdkAppSettings
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.framework.network.getResult
import com.shipsy.ondemand.riderapp.framework.network.model.BaseResponse
import com.shipsy.ondemand.riderapp.interactor.repository.SDKRepository
import com.shipsy.ondemand.riderapp.interactor.usecase.sdk.FetchSdkHealthDataUseCase

/**
 * Created by Kalpesh Kundanani on 26/09/22
 */
class FetchSdkHealthDataUseCaseImpl(
    private val localStore: LocalStore,
    private val repo: SDKRepository
) : FetchSdkHealthDataUseCase {
    override suspend fun invoke(): UseCaseResult<SdkAppSettings?> {
        if (shouldCallFetchHealthCheckInterval()) {
            updateHealthCheckInterval()
            return fetchData()
        }
        return UseCaseResult.success(null)
    }

    private suspend fun fetchData(): UseCaseResult<SdkAppSettings> {
        val result = getResult<BaseResponse<SdkAppSettings>> { repo.checkSdkHealth() }
        return if (result.isSuccess) {
            val data = result.getOrNull()?.data
            if (data != null) {
                saveAppSettings(data)
                UseCaseResult.success(data)
            } else {
                UseCaseResult.failure(
                    ErrorData.GeneralError(
                        displayType = DisplayType.SnackBar("Error fetching data.")
                    )
                )
            }
        } else {
            val error = result.getErrorData()
            (error as? ErrorData.Unauthorised)?.let {
                localStore.clear()
            }
            UseCaseResult.failure(error!!)
        }
    }

    private fun shouldCallFetchHealthCheckInterval(): Boolean {
        val lastCalledTime =
            localStore.getValue(PREF_SDK_APP_SETTINGS_SDK_HEALTH_CHECK_API_CALLING_TIME, 0L)
        val currentTime = DateTime.currentTimeMillis()
        val interval = localStore.getValue(
            PREF_SDK_APP_SETTINGS_SDK_HEALTH_CHECK_API_INTERVAL,
            DEFAULT_SDK_HEALTH_CHECK_API_INTERVAL
        )
        val timeDiff = currentTime - lastCalledTime
        return timeDiff > interval
    }

    private fun updateHealthCheckInterval() {
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_SDK_HEALTH_CHECK_API_CALLING_TIME,
            DateTime.currentTimeMillis()
        )
    }

    private fun saveAppSettings(data: SdkAppSettings) {
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_SDK_HEALTH_CHECK_API_INTERVAL,
            data.sdkHealthCheckAPIInterval
        )
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_UPDATE_FIREBASE_COORDINATES,
            data.updateFirebaseCoordinates
        )
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_FIREBASE_PING_INTERVAL,
            data.firebasePingInterval
        )

        //Mixpanel project token
        localStore.putValue(
            PREF_SDK_MIXPANEL_PROJECT_TOKEN,
            data.mixpanelProjectToken ?: ""
        )
    }
}