package com.shipsy.ondemand.riderapp.domain.usecase.sdk

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.framework.data.sdk.SDKMasterDataFetchResponse
import com.shipsy.ondemand.riderapp.framework.data.sdk.SdkAppSettings
import com.shipsy.ondemand.riderapp.framework.data.sdk.SdkWorkerDetails
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.framework.network.getResult
import com.shipsy.ondemand.riderapp.framework.network.model.BaseResponse
import com.shipsy.ondemand.riderapp.interactor.repository.SDKRepository
import com.shipsy.ondemand.riderapp.interactor.usecase.sdk.FetchMasterDataUseCase

/**
 * Created by Kalpesh Kundanani on 26/09/22
 */
class FetchMasterDataUseCaseImpl(
    private val repo: SDKRepository,
    private val localStore: LocalStore
) : FetchMasterDataUseCase {
    override suspend fun invoke(input: FetchMasterDataUseCase.Input): UseCaseResult<SDKMasterDataFetchResponse> {
        return fetchData(input)
    }

    private suspend fun fetchData(input: FetchMasterDataUseCase.Input): UseCaseResult<SDKMasterDataFetchResponse> {
        localStore.putValue(USER_AUTH_TOKEN, input.authToken)
        localStore.putValue(ORG_ID, input.orgId)
        localStore.putValue(rider_id, input.riderId)
        val result = getResult<BaseResponse<SDKMasterDataFetchResponse>> {
            repo.sdkMasterDataFetch()
        }
        if (result.isSuccess) {
            val data = result.getOrNull()?.data
            if (data != null) {
                data.organisationID?.let {
                    localStore.putValue(ORG_ID, it)
                }

                saveFirebaseData(data)
                data.workerDetails?.let { saveWorkerDetails(it) }
                data.appSettings?.let { saveAppSettings(it) }
                return UseCaseResult.success(data)
            } else {
                return snack("Error Fetching Data")
            }
        }
        return UseCaseResult.failure(result.getErrorData()!!)
    }

    private fun saveFirebaseData(data: SDKMasterDataFetchResponse) {
        localStore.putValue(
            PREF_SDK_FIREBASE_DATA_FIREBASE_APPLICATION_ID,
            data.firebaseApplicationID
        )
        localStore.putValue(PREF_SDK_FIREBASE_DATA_FIREBASE_DATABASE_URL, data.firebaseDatabaseURL)
        localStore.putValue(PREF_SDK_FIREBASE_DATA_FIREBASE_API_KEY, data.firebaseAPIKey)
        localStore.putValue(PREF_SDK_FIREBASE_DATA_FIREBASE_PROJECT_ID, data.firebaseProjectID)
    }

    private fun saveWorkerDetails(details: SdkWorkerDetails) {
        localStore.putValue(worker_code, details.workerCode)
        localStore.putValue(HUB_CODE, details.hubCode)
        localStore.putValue(rider_id, details.workerId)
        localStore.putValue(FIREBASE_TOKEN, details.firebaseAccessToken)
    }

    private fun saveAppSettings(settings: SdkAppSettings) {
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_SDK_HEALTH_CHECK_API_INTERVAL, settings.sdkHealthCheckAPIInterval
        )
        settings.sdkHealthCheckAPIInterval?.let {
            localStore.putValue(
                PREF_SDK_APP_SETTINGS_HEALTH_CHECK_API_CALL_TIME,
                DateTime.currentTimeMillis() + it
            )
        }
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_UPDATE_FIREBASE_COORDINATES, settings.updateFirebaseCoordinates
        )
        localStore.putValue(
            PREF_SDK_APP_SETTINGS_FIREBASE_PING_INTERVAL, settings.firebasePingInterval
        )

        localStore.putValue(
            PREF_SDK_MIXPANEL_PROJECT_TOKEN,
            settings.mixpanelProjectToken ?: "")
    }

    private fun snack(message: String): UseCaseResult<SDKMasterDataFetchResponse> =
        UseCaseResult.failure(
            ErrorData.GeneralError(
                displayType = DisplayType.SnackBar(error = message)
            )
        )
}