package com.shipsy.ondemand.riderapp.domain.usecase.riderstate

import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.cache.dao.mapToOrderData
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.domain.orderstatus.OrderType
import com.shipsy.ondemand.riderapp.firebase.FirebaseEventHandler
import com.shipsy.ondemand.riderapp.framework.data.*
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.IsPudoFlowEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.pickupscan.IsPickingFlowEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.FindOrderDetailUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.OrdersFetchUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.RiderCurrentStateUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.statemanagement.OrderListUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.statemanagement.OrderToShowForDetailUseCase
import com.shipsy.ondemand.riderapp.shared.cache.CurrentOrders

class RiderCurrentStateUseCaseImpl(
    private val localStore: LocalStore,
    private val orderListUseCase: OrderListUseCase,
    private val orderToShowForDetailUseCase: OrderToShowForDetailUseCase,
    private val findOrderDetailUseCase: FindOrderDetailUseCase,
    private val isPudoFlowEnabledUseCase: IsPudoFlowEnabledUseCase,
    private val isPickingFlowEnabledUseCase: IsPickingFlowEnabledUseCase,
    private val ordersFetchUseCase: OrdersFetchUseCase,
) : RiderCurrentStateUseCase {
    override suspend fun invoke(input: RiderCurrentStateUseCase.Input): RiderStatusState {
        return getRiderState(input)
    }

    private suspend fun getRiderState(input: RiderCurrentStateUseCase.Input): RiderStatusState {
        val isCheckIn = localStore.getValue(
            RIDER_STATE_RESPONSE_CHECKED_IN,
            false
        )
        return if (isCheckIn) {
            RiderStatusState.CheckedIn(checkedInState(input))
        } else {
            FirebaseEventHandler.logUiUpdatedEvent("CheckedOut")
            RiderStatusState.CheckedOut
        }
    }

    private suspend fun checkedInState(input: RiderCurrentStateUseCase.Input): CheckedInStatus {
        val workerStatus = localStore.getValue(
            RIDER_STATE_RESPONSE_WORKER_STATUS,
            ""
        )
        val waitingForSlot = localStore.getValue(
            RIDER_STATE_RESPONSE_WORKER_WAITING_STATUS,
            false
        )
        val isOnBreak = localStore.getValue(
            RIDER_STATE_RESPONSE_IS_ON_BREAK,
            false
        )

        val isPudoFlow = isPudoFlowEnabledUseCase.invoke()
        val orderDetailReferenceNumber = orderToShowForDetailUseCase.invoke(
            OrderToShowForDetailUseCase.Input(isPudoFlow, workerStatus),
        )

        if(input.goBack){
          return  orderListScreens(workerStatus)
        }

        return when {
            isOnBreak -> {
                FirebaseEventHandler.logUiUpdatedEvent("OnBreak")
                CheckedInStatus.OnBreak
            }
            waitingForSlot -> {
                FirebaseEventHandler.logUiUpdatedEvent("WaitingForSlot")
                CheckedInStatus.WaitingForSlot
            }
            workerStatus.isNotEmpty() && !orderDetailReferenceNumber.referenceNumber.isNullOrEmpty() -> findOrderDetailScreen(
                orderDetailReferenceNumber.referenceNumber,
                workerStatus,
                orderDetailReferenceNumber.requestOrderDetail,
            )
            else ->
                orderListScreens(workerStatus)

        }

    }

    private suspend fun findOrderDetailScreen(
        referenceNumber: String,
        workerStatus: String,
        requestOrderDetail: Boolean = false
    ): CheckedInStatus {
        val orderDetail = findOrderDetailUseCase.invoke(
            FindOrderDetailUseCase.Input(
                referenceNumber,
                requestOrderDetail
            )
        )
        return orderDetail.checkedInStatus ?: orderListScreens(workerStatus)
    }

    private suspend fun orderListScreens(workerStatus: String): CheckedInStatus {
        val orders = ordersFetchUseCase.invoke()
        val isPudoFlowEnabled = isPudoFlowEnabledUseCase.invoke()
        val status = if (isPudoFlowEnabled && orders.isNotEmpty())  WorkerStatusType.OUT_FOR_DELIVERY else workerStatus
        return when (status) {
            WorkerStatusType.OUT_FOR_DELIVERY -> CheckedInStatus.OutForDeliver(
                WorkerStatus.OutForDeliver, getOutForDeliveryScreen()
            )
            WorkerStatusType.IN_STORE -> CheckedInStatus.EligibleForOrder(
                WorkerStatus.InStore,
                getInStoreScreen()
            )
            WorkerStatusType.RETURNING_TO_STORE -> {
                val currentOrders = getReturningToStoreOrders()
//                FirebaseEventHandler.logUiUpdatedEvent("ReturningToStore", currentOrders)
                val referenceNumber = currentOrders.map { it.reference_number }
                CheckedInStatus.ReturningToStore(referenceNumber)
            }
            else -> {
                FirebaseEventHandler.logUiUpdatedEvent("WaitingForOrder")
                CheckedInStatus.EligibleForOrder(
                    WorkerStatus.InStore,
                    screen = InStoreScreen.WaitingForOrder
                )
            }
        }
    }

    private suspend fun getReturningToStoreOrders() = orderListUseCase.invoke(
        OrderListUseCase.Input(WorkerStatus.ReturningToStore)
    ).currentOrders

    private suspend fun getInStoreScreen(): InStoreScreen {
        val orders = ordersFetchUseCase.invoke()

        // at the time of being in store all orders have same status
        // so, here we are considering only first order.
        val firstOrder = orders.firstOrNull()

        val referenceNumbers = orders.map { it.referenceNumber!! }

        return when (firstOrder?.status) {
            OrderStatusType.ASSIGNED_FOR_DELIVERY -> {
                FirebaseEventHandler.logUiUpdatedEvent( "StartBike", orders)
                InStoreScreen.StartBike(referenceNumbers)
            }
            OrderStatusType.PICKUP_PROCESS_STARTED -> {
                FirebaseEventHandler.logUiUpdatedEvent( "PickupComplete", orders)
                InStoreScreen.PickupComplete(referenceNumbers)
            }
            else -> {
                FirebaseEventHandler.logUiUpdatedEvent("WaitingForOrder")
                InStoreScreen.WaitingForOrder
            }
        }
    }

    private suspend fun getOutForDeliveryScreen(): OutForDeliveryScreen {
        val isPudoFlowEnabled = isPudoFlowEnabledUseCase.invoke()
        val isPickingFlow = isPickingFlowEnabledUseCase.invoke()

        val input = OrderListUseCase.Input(WorkerStatus.OutForDeliver)
        val output = orderListUseCase.invoke(input)

        val filterCompletedPickupOrders: (CurrentOrders) -> Boolean = { it.task_type == OrderType.PICKUP && it.status == OrderStatusType.PICKUP_COMPLETED }

        val currentOrders = output.currentOrders.filterNot(filterCompletedPickupOrders)
        val referenceNumbers = currentOrders.map { it.reference_number }
        return if (isPudoFlowEnabled) {
            val orders = ordersFetchUseCase.invoke()

            if (isPickingFlow) {
                val pickUpScanOrders = orders.filter { it.isPickupTask && it.isPickupProcessStarted }
                if (pickUpScanOrders.isNotEmpty()) {
                    FirebaseEventHandler.logUiUpdatedEvent("PickupScanScreen", pickUpScanOrders)
                    return OutForDeliveryScreen.PickupScanScreen(pickUpScanOrders.mapNotNull { it.referenceNumber })
                }
            }

            val rtoOrders = orders.filter { it.isRTOTask }
            FirebaseEventHandler.logOrderDisplayedEvent(currentOrders.mapToOrderData()?.plus(rtoOrders))

            val rtoReferenceNumbers = rtoOrders.mapNotNull { it.referenceNumber }

//            FirebaseEventHandler.logUiUpdatedEvent("OrderListScreen", currentOrders)
            FirebaseEventHandler.logUiUpdatedEvent("OrderListScreen", rtoOrders)
            OutForDeliveryScreen.OrderListScreen(referenceNumbers + rtoReferenceNumbers)
        } else {
//            FirebaseEventHandler.logUiUpdatedEvent("OnGoingListScreen", currentOrders)
            return OutForDeliveryScreen.OnGoingListScreen(referenceNumbers)
        }
    }

}