package com.shipsy.ondemand.riderapp.domain.usecase.riderstate

import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.interactor.helper.UiRefreshNeeded
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.IsPudoFlowEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.FetchRiderStateFromLocal
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.LocalEventHandlingUseCase

class LocalEventHandlingUseCaseImpl(
    private val database: Database,
    private val localStore: LocalStore,
    private val fetchRiderStateFromLocal: FetchRiderStateFromLocal,
    private val updateUiRefreshNeeded: UiRefreshNeeded,
    private val isPudoFlowEnabledUseCase: IsPudoFlowEnabledUseCase
) : LocalEventHandlingUseCase {
    private val TAG = "LocalEventHandlingUseCaseImpl"
    override suspend fun invoke(input: RiderEvent): UseCaseResult<RiderStateResponse> {
        val orderStatus = findOrderStatusFromEventType(input.type)
        println("$TAG -> orderstatus  $orderStatus")
        if (orderStatus.isNotEmpty()) {
            database.currentOrdersDao.updateOrderStatusByReferenceNumber(
                input.reference_number_array,
                orderStatus
            )
            findWorkerStatus(orderStatus)?.let {
                localStore.putValue(
                    RIDER_STATE_RESPONSE_WORKER_STATUS,
                    it
                )
            }
            updateUiRefreshNeeded.updated()
        }



        return fetchRiderStateFromLocal.invoke()
    }

    private suspend fun findWorkerStatus(orderStatus: String): String? {
        val order = database.currentOrdersDao.getOrderNotByStatus(
            listOf(
                OrderStatusType.ATTEMPTED,
                OrderStatusType.DELIVERED
            )
        )
        return if (orderStatus == OrderStatusType.PICKUP_COMPLETED) {
            WorkerStatusType.OUT_FOR_DELIVERY
        } else if ((order?.size ?: 0) > 0) {
            null
        } else if (isPickupFlow()) {
            WorkerStatusType.IN_STORE
        } else{
            WorkerStatusType.RETURNING_TO_STORE
        }
    }

    private fun isPickupFlow() : Boolean {
        return isPudoFlowEnabledUseCase.invoke()
    }

    private fun findOrderStatusFromEventType(eventName: String): String {
        println("$TAG findOrderStatusFromEventType: eventName: $eventName")
        return when (eventName) {
            RiderStateEventType.REACHED_GATE -> OrderStatusType.REACHED_GATE
            RiderStateEventType.DELIVERED -> OrderStatusType.DELIVERED
            RiderStateEventType.UNDELIVERED -> OrderStatusType.ATTEMPTED
            RiderStateEventType.GOING_FOR_PICKUP -> OrderStatusType.PICKED_UP
            RiderStateEventType.PICKING_PROCESS_STARTED -> OrderStatusType.PICKUP_PROCESS_STARTED
            RiderStateEventType.PICKUP_COMPLETED -> OrderStatusType.PICKUP_COMPLETED
            else -> ""
        }
    }
}