package com.shipsy.ondemand.riderapp.domain.usecase.riderstate

import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.cache.dao.mapListOrderToCurrentOrder
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.framework.network.model.login.CurrentOrder
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderCurrentState
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.FetchRiderStateFromLocal

class FetchRiderStateFromLocalImpl(
    private val database: Database,
    private val localStore: LocalStore
) : FetchRiderStateFromLocal {
    override suspend fun invoke(): UseCaseResult<RiderStateResponse> {
        val riderState = fetchRiderState()
        val currentOrder = fetchCurrentOrder()
        return UseCaseResult.success(
            RiderStateResponse(
                RiderCurrentState(
                    checked_in = riderState.isCheckedIn,
                    worker_status = riderState.workerStatus,
                    waiting_for_slot = riderState.waitingForSlot,
                    is_on_break = riderState.isOnBreak,
                    current_orders = currentOrder,
                    trip_reference_number= riderState.tripReferenceNumber,
                )
            )
        )
    }

    private fun fetchRiderState(): WorkerStatusData {
        val isCheckIn = localStore.getValue(
            RIDER_STATE_RESPONSE_CHECKED_IN,
            false
        )

        val workerStatus = localStore.getValue(
            RIDER_STATE_RESPONSE_WORKER_STATUS,
            ""
        )
        val waitingForSlot = localStore.getValue(
            RIDER_STATE_RESPONSE_WORKER_WAITING_STATUS,
            false
        )
        val isOnBreak = localStore.getValue(
            RIDER_STATE_RESPONSE_IS_ON_BREAK,
            false
        )
        val tripReferenceNumber = localStore.getValue(
            RIDER_STATE_RESPONSE_TRIP_REFERENCE_NUMBER,
            ""
        )
        return WorkerStatusData(isCheckIn, workerStatus, waitingForSlot, isOnBreak, tripReferenceNumber)
    }

    private suspend fun fetchCurrentOrder(): List<CurrentOrder> {
        return database.currentOrdersDao.getCurrentOrders().mapListOrderToCurrentOrder()
    }

}

data class WorkerStatusData(
    val isCheckedIn: Boolean,
    val workerStatus: String,
    val waitingForSlot: Boolean,
    val isOnBreak: Boolean,
    val tripReferenceNumber: String,
)