package com.shipsy.ondemand.riderapp.domain.usecase.pickupprocess

import com.shipsy.ondemand.core.interactor.LocalStore
import com.shipsy.ondemand.riderapp.domain.const.*
import com.shipsy.ondemand.riderapp.framework.network.model.ScanReasonPopupState
import com.shipsy.ondemand.riderapp.framework.network.model.login.HUDetail
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.pickupscan.FetchPickupScanReasonDialogStateUseCase
import kotlinx.serialization.builtins.ListSerializer
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.json.Json

/**
 * Created by Kalpesh Kundanani on 07/09/22
 */
class FetchPickupScanReasonDialogStateUseCaseImpl(
    val localStore: LocalStore
) : FetchPickupScanReasonDialogStateUseCase {
    override fun invoke(input: HUDetail): ScanReasonPopupState? {
        return when {
            input.isStatusManualOrSuccess() -> getOutput(input, MANUAL_PICKUP)
            input.isScanFailed() -> getOutput(input, MANUAL_PICKUP_FAILED)
            input.isEnterManuallyFailed() -> getOutput(input, ENTER_MANUALLY_FAILED)
            else -> getOutput(input, SCAN_FAILED)
        }
    }

    private fun getOutput(input: HUDetail, type: String): ScanReasonPopupState? {
        val isReasonRequired =
            localStore.getValue(PICKUP_CONFIG_FAILURE_REASON_REQUIRED, false)
        val isImageRequired =
            localStore.getValue(PICKUP_CONFIG_FAILURE_IMAGE_REQUIRED, false)
        return when (type) {
            MANUAL_PICKUP -> {
                ScanReasonPopupState(
                    input.hu_number!!,
                    "Manual Pickup",
                    getReason(SHARED_PREF_MANUAL_PICKUP_REASON_LIST),
                    isReasonRequired,
                    isImageRequired,
                )
            }
            SCAN_FAILED -> {
                ScanReasonPopupState(
                    input.hu_number!!,
                    "Scan Failed",
                    getReason(SHARED_PREF_SCAN_FAILED_REASON_LIST),
                    isReasonRequired,
                    isImageRequired,
                )
            }
            MANUAL_PICKUP_FAILED -> {
                ScanReasonPopupState(
                    input.hu_number!!,
                    "Manual Pickup Failed",
                    getReason(SHARED_PREF_NOT_MANUAL_PICKUP_REASON_LIST),
                    isReasonRequired,
                    isImageRequired,
                )
            }
            ENTER_MANUALLY_FAILED -> {
                ScanReasonPopupState(
                    input.hu_number!!,
                    "Manual Entry Failed",
                    getReason(SHARED_PREF_MANUAL_ENTRY_REASON_LIST),
                    isReasonRequired,
                    isImageRequired,
                )
            }
            else -> null
        }
    }

    private fun getReason(key: String) : List<String> {
        val json = localStore.getValue(key, "")
        if(json.isBlank()) return emptyList()
        return Json.decodeFromString(ListSerializer(String.serializer()), json)
    }

    companion object {
        private const val SCAN_FAILED = "SCAN_FAILED"
        private const val MANUAL_PICKUP = "MANUAL_PICKUP"
        private const val MANUAL_PICKUP_FAILED = "MANUAL_DELIVERY_FAILED"
        private const val ENTER_MANUALLY_FAILED = "ENTER_MANUALLY_FAILED"
    }
}