package com.shipsy.ondemand.riderapp.domain.usecase.orderhistory

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.mapper.mapToOrderHistoryList
import com.shipsy.ondemand.riderapp.framework.data.orderhistory.OrderHistoryData
import com.shipsy.ondemand.riderapp.framework.data.orderhistory.OrdersOfDate
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.core.framework.network.getResult
import com.shipsy.ondemand.riderapp.framework.network.model.BaseResponse
import com.shipsy.ondemand.riderapp.interactor.repository.HistoryRepository
import com.shipsy.ondemand.riderapp.interactor.usecase.orderhistory.FetchOrderByDateUseCase
import com.shipsy.ondemand.riderapp.shared.cache.OrderHistory

/**
 * Created by Kalpesh Kundanani on 22/08/22
 */
class FetchOrderByDateUseCaseImpl(
    private val repo: HistoryRepository,
    private val database: Database
) : FetchOrderByDateUseCase {
    override suspend fun invoke(input: Long): UseCaseResult<List<OrderHistory>> {
        return getOrderHistoryData(input)
    }

    private suspend fun getOrderHistoryData(mills: Long): UseCaseResult<List<OrderHistory>> {
        val isToday = DateTime.isToday(mills)
        val date = DateTime.format(mills, "dd-MM-yyyy")
        val ordersByDate: List<OrderHistory> = database.orderHistoryDao.getOrderByDate(date)

        if (!isToday && ordersByDate.isNotEmpty()) {
            return UseCaseResult.success(ordersByDate)
        }

        val res = getResult<BaseResponse<OrderHistoryData>> { repo.getOrderHistory(date, date) }
        if (res.isSuccess) {
            val data: OrdersOfDate? = res.getOrNull()?.data?.order_data?.filterNotNull()?.firstOrNull()
            if (data != null) {
                val orderHistory: List<OrderHistory> = if (!isToday) {
                    database.orderHistoryDao.insertOrderHistory(data)
                    database.orderHistoryDao.getOrderByDate(date)
                } else {
                    data.mapToOrderHistoryList()
                }
                if (orderHistory.isNotEmpty()) {
                    return UseCaseResult.success(orderHistory)
                }
            }
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = true, displayType = DisplayType.SnackBar(error = "No orders found for the given date!")))
        }
        return UseCaseResult.failure(res.getErrorData()!!)
    }
}