package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.domain.const.GeofenceConfigType
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.riderapp.domain.const.TAG_PROOF_OF_DELIVERY
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.HUDetail
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.login.UndeliveredReason
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.UnDeliveredEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.file.UploadFileUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.RecordDeliveryTimeUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class UnDeliveredEventUseCaseImpl(
    private val suspiciousCheckUseCase: SuspiciousCheckUseCase,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase,
    private val uploadFileUseCase: UploadFileUseCase,
    private val recordDeliveryTime: RecordDeliveryTimeUseCase,
) : UnDeliveredEventUseCase {

    private lateinit var input: UnDeliveredEventUseCase.Input
    private var undeliveredReason: UndeliveredReason? = null

    override suspend fun invoke(input: UnDeliveredEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        this.input = input
        var moveForward = geofenceCheck()
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("geofence check failed")))
        }
        moveForward = pickUndeliveredReason()
        if(!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Canceled")))
        }
        return markUnDelivered(input)
    }

    private suspend fun pickUndeliveredReason(): Boolean {
        return suspendCoroutine { continuation ->
            input.handleUndeliveredReasonSelection { moveForward, reason ->
                if (moveForward) {
                    undeliveredReason = reason
                }
                continuation.resume(moveForward)
            }
        }
    }

    private suspend fun geofenceCheck(): Boolean {
        val suspicious =
            suspiciousCheckUseCase.invoke(
                SuspiciousCheckUseCase.Input(
                    GeofenceConfigType.UNDELIVERED,
                    SuspiciousCheckUseCase.LocationCheckFrom.Order(referenceNumber = input.referenceNumber)
                )
            )
        return if (suspicious.showWarning) {
            suspendCoroutine { continuation ->
                input.handleGeofenceError(suspicious.blockOrderProcessing) {
                    continuation.resume(it)
                }
            }
        } else
            true
    }

    private suspend fun markUnDelivered(input: UnDeliveredEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        recordDeliveryTime.invoke(RecordDeliveryTimeUseCase.Input(DateTime.currentTimeMillis()))
        val location = fetchSavedLocation.invoke()
        uploadImagesFromHU(input.huDetails)
        val riderEvent = RiderEvent(
            RiderStateEventType.UNDELIVERED,
            listOf(input.referenceNumber),
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            hu_details = input.huDetails,
            reason = undeliveredReason,
            auto_swipe = false
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

    private suspend fun uploadImagesFromHU(huDetails: List<HUDetail>?) {
        huDetails?.forEach { detail ->
            detail.imageAsBytes?.let {
                detail.image = uploadFileUseCase.invoke(
                    UploadFileUseCase.Input(
                    TAG_PROOF_OF_DELIVERY, it
                ))
                detail.imageAsBytes = null
            }
        }
    }

}