package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.framework.network.model.login.GeoConfigModel
import com.shipsy.ondemand.riderapp.interactor.usecase.FetchHubDetailUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.GetGeoConfigUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.CalculateDistanceBetweenLocationUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation

class SuspiciousCheckUseCaseImpl(
    private val fetchSavedLocation: FetchSavedLocation,
    private val fetchHubDetailUseCase: FetchHubDetailUseCase,
    private val database: Database,
    private val getGeoConfigUseCase: GetGeoConfigUseCase,
    private val calculateDistanceBetweenLocationUseCase: CalculateDistanceBetweenLocationUseCase
) : SuspiciousCheckUseCase {
    override suspend fun invoke(input: SuspiciousCheckUseCase.Input): SuspiciousCheckUseCase.Output {
        return checkGeofenceConfig(
            input.checkWith,
            getGeoConfigUseCase.invoke(GetGeoConfigUseCase.Input(input.eventType)).geoConfigModel
        )

    }

    private suspend fun checkGeofenceConfig(
        checkWith: SuspiciousCheckUseCase.LocationCheckFrom,
        geoConfigModel: GeoConfigModel
    ): SuspiciousCheckUseCase.Output {
        if (!geoConfigModel.display_warning && !geoConfigModel.geofence)
            return getDefaultResponse()
        val location = fetchSavedLocation.invoke()
        val origin = when (checkWith) {
            SuspiciousCheckUseCase.LocationCheckFrom.Hub -> {
                val hubDetail = fetchHubDetailUseCase.invoke()
                Pair(hubDetail.lat, hubDetail.lng)
            }
            is SuspiciousCheckUseCase.LocationCheckFrom.Order ->
                fetchOrderLocation(checkWith.referenceNumber)
        }
        val distance =calculateDistanceBetweenLocationUseCase.invoke(
            CalculateDistanceBetweenLocationUseCase.Input(origin.first, origin.second, location.lat, location.lng)).distance

        val thresholdDistance = geoConfigModel.geofence_distance ?: 0
        println("suspicious check ==> distance $distance threshold $thresholdDistance")
        return if (distance > thresholdDistance) {
            println(
                "geo return 5 ${geoConfigModel.display_warning},\n" +
                        "                ${geoConfigModel.geofence}"
            )

            SuspiciousCheckUseCase.Output(
                geoConfigModel.display_warning,
                geoConfigModel.geofence
            )
        } else {
            println("geo return 4")

            getDefaultResponse()
        }

    }

    private suspend fun fetchOrderLocation(referenceNumber: String): Pair<Double?, Double?> {
        val order = database.currentOrdersDao.getOrdersByReferenceNumber(referenceNumber)?.location
        return Pair(order?.lat, order?.lng)
    }

    private fun getDefaultResponse() =
        SuspiciousCheckUseCase.Output(showWarning = false, blockOrderProcessing = false)

}