package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.domain.const.GeofenceConfigType
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.ReturnToStoreEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class ReturnToStoreEventUseCaseImpl(
    private val suspiciousCheckUseCase: SuspiciousCheckUseCase,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase
) : ReturnToStoreEventUseCase {

    private lateinit var input: ReturnToStoreEventUseCase.Input

    override suspend fun invoke(input: ReturnToStoreEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        this.input = input
        val moveForward = geofenceCheck()
        if (!moveForward) {
            return UseCaseResult.failure(
                ErrorData.GeneralError(
                    isVisible = false,
                    DisplayType.SnackBar("geofence check failed")
                )
            )
        }
        return markBackToStore(input)
    }

    private suspend fun geofenceCheck(): Boolean {
        val suspicious =
            suspiciousCheckUseCase.invoke(
                SuspiciousCheckUseCase.Input(
                    GeofenceConfigType.BACK_TO_STORE,
                    SuspiciousCheckUseCase.LocationCheckFrom.Hub
                )
            )
        return if (suspicious.showWarning) {
            suspendCoroutine { continuation ->
                input.handleGeofenceError?.let {
                    it(suspicious.blockOrderProcessing) {
                        continuation.resume(it)
                    }
                } ?: continuation.resume(false)
            }
        } else
            true
    }

    private suspend fun markBackToStore(input: ReturnToStoreEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        val location = fetchSavedLocation.invoke()
        val riderEvent = RiderEvent(
            RiderStateEventType.RETURN_TO_HUB,
            input.referenceNumbers,
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            auto_swipe = input.autoSwipe
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

}