package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.domain.const.GeofenceConfigType
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.ReachGateEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.RecordReachGateTimeUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class ReachGateEventUseCaseImpl(
    private val suspiciousCheckUseCase: SuspiciousCheckUseCase,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase,
    private val recordReachGateTimeUseCase: RecordReachGateTimeUseCase
) : ReachGateEventUseCase {

    private lateinit var input: ReachGateEventUseCase.Input

    override suspend fun invoke(input: ReachGateEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        println("ReachGateEventUseCaseImpl")
        this.input = input

        if (input.autoSwipe) {
            val referenceNumbers = input.referenceNumbers.filter { geofenceCheck(it) }
            return if (referenceNumbers.isEmpty()) geoFenceError()
            else markReachGate(input.copy(referenceNumbers = referenceNumbers))
        } else {
            input.referenceNumbers.forEach {
                val moveForward = geofenceCheck(it)
                if (!moveForward) {
                    return geoFenceError()
                }
            }
            return markReachGate(input)
        }
    }

    private fun geoFenceError(): UseCaseResult<RiderStateResponse> =
        UseCaseResult.failure(
            ErrorData.GeneralError(
                isVisible = false,
                displayType = DisplayType.SnackBar("geofence check failed")
            )
        )


    private suspend fun geofenceCheck(referenceNumber: String): Boolean {
        val suspicious =
            suspiciousCheckUseCase.invoke(
                SuspiciousCheckUseCase.Input(
                    GeofenceConfigType.REACHED_GATE,
                    SuspiciousCheckUseCase.LocationCheckFrom.Order(referenceNumber = referenceNumber)
                )
            )
        return if (suspicious.showWarning && input.handleGeofenceError != null) {
            suspendCoroutine { continuation ->
                input.handleGeofenceError?.invoke(suspicious.blockOrderProcessing) {
                    continuation.resume(it)
                }
            }
        } else
            true
    }

    private suspend fun markReachGate(input: ReachGateEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        recordTime(input)
        val location = fetchSavedLocation.invoke()
        val riderEvent = RiderEvent(
            RiderStateEventType.REACHED_GATE,
            input.referenceNumbers,
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            auto_swipe = input.autoSwipe
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

    private fun recordTime(input: ReachGateEventUseCase.Input) {
        input.referenceNumbers.forEach {
            recordReachGateTimeUseCase.invoke(
                RecordReachGateTimeUseCase.Input(
                    referenceNumber = it,
                    timestamp = DateTime.currentTimeMillis()
                )
            )
        }
    }

}