package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.cache.Database
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.riderapp.domain.const.TAG_PROOF_OF_DELIVERY
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.HUDetail
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.CheckAllowDeliveryOTPValidationUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.PickupCompletedEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.file.UploadFileUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class PickupCompletedEventUseCaseImpl(
    private val database: Database,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase,
    private val uploadFileUseCase: UploadFileUseCase,
    private val checkAllowDeliveryOTPValidationUseCase: CheckAllowDeliveryOTPValidationUseCase,
) : PickupCompletedEventUseCase {

    override suspend fun invoke(input: PickupCompletedEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        val moveForward = otpValidate(input)

        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Otp input failed")))
        }

        return markPickupCompleted(input)
    }

    private suspend fun otpValidate(input: PickupCompletedEventUseCase.Input): Boolean {
        val isOtpRequired = checkAllowDeliveryOTPValidationUseCase.invoke()
        if(!isOtpRequired) {
            return true
        }
        return suspendCoroutine { continuation ->
            input.handleOtpInput {
                continuation.resume(true)
            }
        }
    }

    private suspend fun markPickupCompleted(input: PickupCompletedEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        val location = fetchSavedLocation.invoke()
        val huDetails = input.huDetails
        val orderNumbers = database.currentOrdersDao.getOrdersByReferenceNumbers(input.referenceNumbers)?.map { it.order_number }
        val referenceNumbers = orderNumbers?.let { database.currentOrdersDao.getReferenceNumbersByOrderNumbers(it) } ?: input.referenceNumbers
        uploadImagesFromHU(huDetails)
        val riderEvent = RiderEvent(
            RiderStateEventType.PICKUP_COMPLETED,
            referenceNumbers,
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            hu_details= huDetails,
            auto_swipe= false
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

    private suspend fun uploadImagesFromHU(huDetails: List<HUDetail>?) {
        huDetails?.forEach { detail ->
            detail.imageAsBytes?.let {
                detail.image = uploadFileUseCase.invoke(
                    UploadFileUseCase.Input(
                    TAG_PROOF_OF_DELIVERY, it
                ))
                detail.imageAsBytes = null
            }
        }
    }

}