package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.domain.const.GeofenceConfigType
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.PickingProcessStartedEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.FetchReferenceNumbersByOrderNumberUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class PickingProcessStartedEventUseCaseImpl(
    private val suspiciousCheckUseCase: SuspiciousCheckUseCase,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase,
    private val fetchReferenceNumbers: FetchReferenceNumbersByOrderNumberUseCase,
) : PickingProcessStartedEventUseCase {

    private lateinit var input: PickingProcessStartedEventUseCase.Input

    override suspend fun invoke(input: PickingProcessStartedEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        this.input = input

        val moveForward = geofenceCheck()
        if (!moveForward) {
            return UseCaseResult.failure(
                ErrorData.GeneralError(
                    isVisible = false,
                    DisplayType.SnackBar("geofence check failed")
                )
            )
        }

        return markPickingProcessStarted(input)
    }

    private suspend fun geofenceCheck(): Boolean {
        val suspicious =
            suspiciousCheckUseCase.invoke(
                SuspiciousCheckUseCase.Input(
                    GeofenceConfigType.PICKING_UP_STARTED,
                    SuspiciousCheckUseCase.LocationCheckFrom.Hub
                )
            )
        return if (suspicious.showWarning) {
            suspendCoroutine { continuation ->
                input.handleGeofenceError?.let {
                    it(suspicious.blockOrderProcessing) {
                        continuation.resume(it)
                    }
                } ?: continuation.resume(false)
            }
        } else
            true
    }

    private suspend fun markPickingProcessStarted(input: PickingProcessStartedEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        val refNumbers =
            input.referenceNumbers ?: fetchReferenceNumbers.invoke(listOf(input.orderNumber!!))
        val location = fetchSavedLocation.invoke()
        val riderEvent = RiderEvent(
            RiderStateEventType.PICKING_PROCESS_STARTED,
            refNumbers,
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            auto_swipe = input.autoSwipe
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

}