package com.shipsy.ondemand.riderapp.domain.usecase.eventhandler

import com.shipsy.ondemand.core.framework.datetime.DateTime
import com.shipsy.ondemand.riderapp.domain.const.GeofenceConfigType
import com.shipsy.ondemand.riderapp.domain.const.RiderStateEventType
import com.shipsy.ondemand.riderapp.domain.const.TAG_PROOF_OF_DELIVERY
import com.shipsy.ondemand.riderapp.framework.data.oderdata.OrderData
import com.shipsy.ondemand.core.framework.network.DisplayType
import com.shipsy.ondemand.core.framework.network.ErrorData
import com.shipsy.ondemand.core.framework.network.UseCaseResult
import com.shipsy.ondemand.riderapp.framework.network.model.login.HUDetail
import com.shipsy.ondemand.riderapp.framework.network.model.login.PaymentDetails
import com.shipsy.ondemand.riderapp.framework.network.model.login.RelationData
import com.shipsy.ondemand.riderapp.framework.network.model.login.RiderStateResponse
import com.shipsy.ondemand.riderapp.framework.network.model.riderevent.RiderEvent
import com.shipsy.ondemand.riderapp.framework.uuid.Uuid
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.CheckAllowDeliveryOTPValidationUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.IsReceiverRelationEnabledUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.appsettings.deliveryscan.CheckShowDeliveryScan
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.DeliveredEventUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.eventhandler.SuspiciousCheckUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.file.UploadFileUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.location.FetchSavedLocation
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.OrderDetailFetchUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.orderdetail.RecordDeliveryTimeUseCase
import com.shipsy.ondemand.riderapp.interactor.usecase.riderstate.EventSyncUseCase
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class DeliveredEventUseCaseImpl(
    private val uploadFileUseCase: UploadFileUseCase,
    private val checkShowDeliveryScan: CheckShowDeliveryScan,
    private val orderDetailFetchUseCase: OrderDetailFetchUseCase,
    private val suspiciousCheckUseCase: SuspiciousCheckUseCase,
    private val fetchSavedLocation: FetchSavedLocation,
    private val eventSyncUseCase: EventSyncUseCase,
    private val checkAllowDeliveryOTPValidationUseCase: CheckAllowDeliveryOTPValidationUseCase,
    private val recordDeliveryTime: RecordDeliveryTimeUseCase,
    private val isReceiverRelationEnabled: IsReceiverRelationEnabledUseCase,
) : DeliveredEventUseCase {

    private lateinit var input: DeliveredEventUseCase.Input
    private var paymentDetails: List<PaymentDetails>? = null
    private var receiverDetails: RelationData? = null
    private var huDetails: List<HUDetail>? = null

    override suspend fun invoke(input: DeliveredEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        this.input = input

        val orderData =
            orderDetailFetchUseCase.invoke(OrderDetailFetchUseCase.Input(input.referenceNumber))

        var moveForward = deliveryScan(orderData)
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Delivery scan failed")))
        }

        moveForward = geofenceCheck()
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("geofence check failed")))
        }

        moveForward = pickReceiverDetails()
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Receiver input failed")))
        }

        moveForward = otpValidate(orderData)
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Otp input failed")))
        }

        moveForward = pickPaymentDetails(orderData)
        if (!moveForward) {
            return UseCaseResult.failure(ErrorData.GeneralError(isVisible = false, DisplayType.SnackBar("Payment input failed")))
        }

        return markDelivered(input)
    }

    private suspend fun deliveryScan(orderData: OrderData): Boolean {
        val isDeliveryScan = checkShowDeliveryScan.invoke(orderData)
        if(!isDeliveryScan) {
            return true
        }
        return suspendCoroutine { continuation ->
            input.handleDeliveryScan {
                huDetails = it
                continuation.resume(true)
            }
        }
    }

    private suspend fun otpValidate(orderData: OrderData): Boolean {
        if (!orderData.isDeliveryTask) {
            return true
        }
        val isOtpRequired = checkAllowDeliveryOTPValidationUseCase.invoke()
        if(!isOtpRequired) {
            return true
        }
        return suspendCoroutine { continuation ->
            input.handleOtpInput {
                continuation.resume(true)
            }
        }
    }

    private suspend fun pickPaymentDetails(orderData: OrderData): Boolean {
        val amount = orderData.amount ?: 0.0
        if(amount <= 0) {
            return true
        }
        return suspendCoroutine { continuation ->
            input.handlePaymentDetailInput { details ->
                paymentDetails = listOf(details)
                continuation.resume(true)
            }
        }
    }

    private suspend fun pickReceiverDetails(): Boolean {
        if (!isReceiverRelationEnabled.invoke()) return true
        return suspendCoroutine { continuation ->
            input.handleReceiverRelationInput { details ->
                receiverDetails = details
                continuation.resume(true)
            }
        }
    }

    private suspend fun geofenceCheck(): Boolean {
        val suspicious =
            suspiciousCheckUseCase.invoke(
                SuspiciousCheckUseCase.Input(
                    GeofenceConfigType.DELIVERED,
                    SuspiciousCheckUseCase.LocationCheckFrom.Order(referenceNumber = input.referenceNumber)
                )
            )
        return if (suspicious.showWarning) {
            suspendCoroutine { continuation ->
                input.handleGeofenceError(suspicious.blockOrderProcessing) {
                    continuation.resume(it)
                }
            }
        } else
            true
    }

    private suspend fun markDelivered(input: DeliveredEventUseCase.Input): UseCaseResult<RiderStateResponse> {
        recordDeliveryTime.invoke(RecordDeliveryTimeUseCase.Input(DateTime.currentTimeMillis()))
        val location = fetchSavedLocation.invoke()
        uploadImagesFromHU(huDetails)
        val riderEvent = RiderEvent(
            RiderStateEventType.DELIVERED,
            listOf(input.referenceNumber),
            Uuid().generateUuid(),
            DateTime.currentTimeMillis(),
            location.lat,
            location.lng,
            payment_details = paymentDetails,
            auto_swipe = false,
            hu_details = huDetails,
            receiver_relation = receiverDetails,
        )
        return eventSyncUseCase.invoke(EventSyncUseCase.Input(input.syncType, riderEvent))
    }

    private suspend fun uploadImagesFromHU(huDetails: List<HUDetail>?) {
        huDetails?.forEach { detail ->
            detail.imageAsBytes?.let {
                detail.image = uploadFileUseCase.invoke(UploadFileUseCase.Input(
                    TAG_PROOF_OF_DELIVERY, it
                ))
                detail.imageAsBytes = null
            }
        }
    }

}